<?php

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\ProductController;
use App\Http\Controllers\CheckoutController;
use App\Http\Controllers\PaymentController;
use App\Http\Controllers\DealsController;
use App\Http\Controllers\CartController;
use App\Models\Product;
use App\Models\Category;
use App\Models\Deal;
use App\Models\Order;

/*
|--------------------------------------------------------------------------
| Web Routes
|--------------------------------------------------------------------------
*/

// PUBLIC ROUTES (No authentication required)
Route::get('/', function () {
    // Get featured products (active and in stock)
    $featuredProducts = Product::where('is_active', true)
        ->where('stock', '>', 0)
        ->orderBy('created_at', 'desc')
        ->take(12)
        ->get();
    
    // Get active categories with product counts
    $categories = Category::where('is_active', true)
        ->whereNull('parent_id')
        ->withCount(['products' => function ($query) {
            $query->where('is_active', true);
        }])
        ->get();
    
    // Get featured deals
    $featuredDeals = Deal::with(['products' => function($query) {
            $query->where('is_active', true)
                  ->where('stock', '>', 0)
                  ->limit(4);
        }])
        ->where('is_active', true)
        ->where('featured', true)
        ->where('start_date', '<=', now())
        ->where('end_date', '>=', now())
        ->orderBy('sort_order', 'asc')
        ->limit(3)
        ->get();
    
    return view('welcome', compact('featuredProducts', 'categories', 'featuredDeals'));
})->name('home');

// Public product viewing
Route::get('/products', [ProductController::class, 'publicIndex'])->name('products.public');
Route::get('/products/{product}', [ProductController::class, 'publicShow'])->name('products.public.show');

// Categories route
Route::get('/categories', function () {
    $categories = Category::where('is_active', true)
        ->whereNull('parent_id')
        ->withCount(['products' => function ($query) {
            $query->where('is_active', true);
        }])
        ->get();
    
    return view('categories', compact('categories'));
})->name('categories.index');

// Category products page
Route::get('/category/{category}/products', function (\App\Models\Category $category) {
    // Get products in this category
    $products = \App\Models\Product::where('is_active', true)
        ->where('category_id', $category->id)
        ->orderBy('created_at', 'desc')
        ->paginate(12);
    
    return view('category-products', [
        'category' => $category,
        'products' => $products
    ]);
})->name('category.products');

// ============ CART ROUTES ============
// Using CartController for persistent cart storage
Route::post('/cart/add', [CartController::class, 'add'])->name('cart.add');
Route::get('/cart/count', [CartController::class, 'count'])->name('cart.count');
Route::get('/cart', [CartController::class, 'view'])->name('cart.view');
Route::post('/cart/remove/{id}', [CartController::class, 'remove'])->name('cart.remove');
Route::post('/cart/update/{id}', [CartController::class, 'update'])->name('cart.update');
Route::post('/cart/clear', [CartController::class, 'clear'])->name('cart.clear');
Route::post('/cart/merge', [CartController::class, 'mergeGuestCart'])->name('cart.merge');
// ============ END CART ROUTES ============


Route::post('/store-redirect-url', function (\Illuminate\Http\Request $request) {
    $request->validate(['redirect_url' => 'required|url']);
    session()->put('redirect_url', $request->redirect_url);
    return response()->json(['success' => true]);
});

// ============ M-PESA PAYMENT ROUTES ============

// Protected checkout routes (requires authentication)
Route::middleware('auth')->group(function () {
    Route::get('/checkout', function () {
        // Use CartController logic to get cart items
        if (auth()->check()) {
            // Get from database for logged-in users
            $cartItems = \App\Models\Cart::with('product')
                ->where('user_id', auth()->id())
                ->get();
        } else {
            // Get from session for guests (though middleware should prevent this)
            $sessionCart = session()->get('cart', []);
            $cartItems = collect();
            
            foreach ($sessionCart as $item) {
                $product = \App\Models\Product::find($item['product_id']);
                if ($product) {
                    $cartItems->push((object)[
                        'product' => $product,
                        'quantity' => $item['quantity']
                    ]);
                }
            }
        }
        
        if ($cartItems->count() == 0) {
            return redirect()->route('cart.view')->with('error', 'Your cart is empty');
        }
        
        // Calculate totals
        $subtotal = 0;
        foreach ($cartItems as $item) {
            $subtotal += $item->product->price * $item->quantity;
        }
        
       // $tax = $subtotal * 0.16; // 16% VAT for Kenya
        //$shipping = $subtotal > 1000 ? 0 : 200; // Free shipping above 1000
        //$total = $subtotal + $tax + $shipping;

         $tax = 0;
        $shipping = 0;
        $total = $subtotal;
        
        // Store totals in session for later use
        session()->put('cart_subtotal', $subtotal);
        session()->put('cart_total', $total);
        
        return view('checkout', [
            'cartItems' => $cartItems,
            'subtotal' => $subtotal,
            'total' => $total,
            'tax' => $tax,
            'shipping' => $shipping,
        ]);
    })->name('checkout');
    
    // M-Pesa Payment Routes (also require auth)
    Route::post('/process-mpesa-payment', [PaymentController::class, 'processMpesaPayment'])->name('process.mpesa.payment');
    Route::get('/check-payment-status', [PaymentController::class, 'checkPaymentStatus'])->name('check.payment.status');
    Route::post('/process-checkout', [PaymentController::class, 'processCheckout'])->name('checkout.process');
    Route::post('/process-cod-checkout', [CheckoutController::class, 'processCODCheckout'])->name('checkout.process.cod');
    Route::post('/process-paybill-checkout', [CheckoutController::class, 'processPaybillCheckout'])->name('checkout.process.paybill');
    Route::get('/order-confirmation/{order}', [CheckoutController::class, 'orderConfirmation'])->name('order.confirmation');
});

// M-Pesa Callback URL (Webhook) - Called by Safaricom (doesn't need auth)
Route::post('/mpesa-callback', [PaymentController::class, 'mpesaCallback'])->name('mpesa.callback');

// Store payment phone temporarily (public)
Route::post('/store-payment-phone', function (\Illuminate\Http\Request $request) {
    $request->validate(['phone' => 'required|string']);
    session()->put('payment_phone', $request->phone);
    return response()->json(['success' => true]);
});

// Test route for M-Pesa (remove in production) - public
Route::get('/test-mpesa', [PaymentController::class, 'testPaymentPage'])->name('test.mpesa');

// ============ END M-PESA ROUTES ============

// PUBLIC DEALS ROUTES
Route::get('/deals', [DealsController::class, 'index'])->name('deals.index');
Route::get('/deals/{slug}', [DealsController::class, 'show'])->name('deals.show');
Route::post('/deals/{id}/apply', [DealsController::class, 'apply'])->name('deals.apply');
Route::post('/deals/remove', [DealsController::class, 'remove'])->name('deals.remove');

// Authentication routes (Login/Register)
require __DIR__.'/auth.php';

// Transaction routes (protected - for logged in users)
Route::middleware('auth')->group(function () {
    Route::get('/my-transactions', [PaymentController::class, 'myTransactions'])->name('transactions.index');
    Route::get('/transactions/{id}', [PaymentController::class, 'viewTransaction'])->name('transactions.show');
});

// ADMIN ROUTES (Authentication AND Admin role required)
Route::middleware(['auth', 'admin'])->group(function () {
    // Admin Dashboard
Route::get('/admin/dashboard', function () {
    // Get Mpesa transaction statistics
    $totalTransactions = \App\Models\MpesaTransaction::count();
    $successfulTransactions = \App\Models\MpesaTransaction::where('status', 'success')->count();
    $totalRevenue = \App\Models\MpesaTransaction::where('status', 'success')->sum('amount');
    $recentTransactions = \App\Models\MpesaTransaction::with(['order', 'user'])
        ->latest()
        ->take(5)
        ->get();
    
    // Get order statistics
    $totalOrders = \App\Models\Order::count();
    $totalProducts = \App\Models\Product::count();
    $totalCustomers = \App\Models\User::count();
    
    return view('admin.dashboard', compact(
        'totalTransactions',
        'successfulTransactions',
        'totalRevenue',
        'recentTransactions',
        'totalOrders',
        'totalProducts',
        'totalCustomers'
    ));
})->name('admin.dashboard');
    // Admin User Management
    Route::resource('admin/users', \App\Http\Controllers\Admin\UserController::class)->names([
        'index' => 'admin.users.index',
        'create' => 'admin.users.create',
        'store' => 'admin.users.store',
        'show' => 'admin.users.show',
        'edit' => 'admin.users.edit',
        'update' => 'admin.users.update',
        'destroy' => 'admin.users.destroy'
    ]);
    
    // Admin Products Management (CRUD)
    Route::resource('admin/products', ProductController::class)->names([
        'index' => 'admin.products.index',
        'create' => 'admin.products.create',
        'store' => 'admin.products.store',
        'show' => 'admin.products.show',
        'edit' => 'admin.products.edit',
        'update' => 'admin.products.update',
        'destroy' => 'admin.products.destroy'
    ]);
    
    // Admin Categories Management (CRUD) - NEW
    Route::resource('admin/categories', \App\Http\Controllers\Admin\CategoryController::class)->names([
        'index' => 'admin.categories.index',
        'create' => 'admin.categories.create',
        'store' => 'admin.categories.store',
        'show' => 'admin.categories.show',
        'edit' => 'admin.categories.edit',
        'update' => 'admin.categories.update',
        'destroy' => 'admin.categories.destroy'
    ]);
    
    // Admin Orders Management (CRUD) - NEW
    Route::resource('admin/orders', \App\Http\Controllers\Admin\OrderController::class)->names([
        'index' => 'admin.orders.index',
        'show' => 'admin.orders.show',
        'edit' => 'admin.orders.edit',
        'update' => 'admin.orders.update',
        'destroy' => 'admin.orders.destroy'
    ])->except(['create', 'store']);

    // Add this in your admin routes section
Route::get('/admin/orders/{order}/invoice', function(Order $order) {
    $order->load(['items.product']);
    return view('admin.orders.invoice', compact('order'));
})->name('admin.orders.invoice');
    
    // Admin M-Pesa Transactions (CRUD) - NEW
    Route::get('/admin/transactions', function () {
        $transactions = \App\Models\MpesaTransaction::orderBy('created_at', 'desc')->paginate(20);
        return view('admin.transactions.index', compact('transactions'));
    })->name('admin.transactions.index');
    
    Route::get('/admin/transactions/{id}', function ($id) {
        $transaction = \App\Models\MpesaTransaction::findOrFail($id);
        return view('admin.transactions.show', compact('transaction'));
    })->name('admin.transactions.show');
    
    // Admin Deals Routes
    Route::prefix('admin')->name('admin.')->group(function () {
        Route::get('/deals', [DealsController::class, 'adminIndex'])->name('deals.index');
        Route::get('/deals/create', [DealsController::class, 'create'])->name('deals.create');
        Route::post('/deals', [DealsController::class, 'store'])->name('deals.store');
        Route::get('/deals/{deal}/edit', [DealsController::class, 'edit'])->name('deals.edit');
        Route::put('/deals/{deal}', [DealsController::class, 'update'])->name('deals.update');
        Route::delete('/deals/{deal}', [DealsController::class, 'destroy'])->name('deals.destroy');
        Route::post('/deals/{deal}/toggle-status', [DealsController::class, 'toggleStatus'])->name('deals.toggle-status');
    });
    
    // Admin Settings Routes
    Route::get('/admin/settings', [\App\Http\Controllers\Admin\SettingsController::class, 'index'])->name('admin.settings');
    Route::post('/admin/settings/profile', [\App\Http\Controllers\Admin\SettingsController::class, 'updateProfile'])->name('admin.settings.profile.update');
    Route::post('/admin/settings/password', [\App\Http\Controllers\Admin\SettingsController::class, 'updatePassword'])->name('admin.settings.password.update');
    Route::post('/admin/settings/system', [\App\Http\Controllers\Admin\SettingsController::class, 'updateSystem'])->name('admin.settings.system.update');
    Route::post('/admin/orders/{order}/verify-paybill', [CheckoutController::class, 'verifyPaybillPayment'])->name('checkout.verify-paybill');
    Route::post('/admin/orders/{order}/reject-paybill', [CheckoutController::class, 'rejectPaybillPayment'])->name('checkout.reject-paybill');

    // Fallback admin route
    Route::get('/dashboard', function () {
        return redirect()->route('admin.dashboard');
    });

// Test email route (temporary - remove after testing)
Route::get('/test-email', function() {
    try {
        Mail::raw('Test email from Finepoint Electronics!', function($message) {
            $message->to('denzykim4@gmail.com')
                    ->subject('Test Email');
        });
        return 'Email sent successfully!';
    } catch (\Exception $e) {
        return 'Error: ' . $e->getMessage();
    }
});

});