@extends('layouts.app')

@section('content')
<div class="container mx-auto px-4 py-8">
    <div class="max-w-md mx-auto">
        <h1 class="text-2xl font-bold mb-6 text-center">Test M-Pesa Integration</h1>
        
        <div class="bg-white rounded-lg shadow-sm p-6 mb-6">
            <h2 class="text-lg font-semibold mb-4">Configuration Status</h2>
            <div class="space-y-2 text-sm">
                <div class="flex justify-between">
                    <span class="text-gray-600">Environment:</span>
                    <span class="font-medium {{ config('services.mpesa.environment') === 'sandbox' ? 'text-green-600' : 'text-blue-600' }}">
                        {{ config('services.mpesa.environment') }}
                    </span>
                </div>
                <div class="flex justify-between">
                    <span class="text-gray-600">Shortcode:</span>
                    <span class="font-medium">{{ config('services.mpesa.shortcode') }}</span>
                </div>
                <div class="flex justify-between">
                    <span class="text-gray-600">Callback URL:</span>
                    <span class="font-medium text-xs">{{ config('services.mpesa.callback_url') }}</span>
                </div>
                <div class="flex justify-between">
                    <span class="text-gray-600">Status:</span>
                    <span class="font-medium text-green-600">✓ Configured</span>
                </div>
            </div>
        </div>
        
        <div class="bg-white rounded-lg shadow-sm p-6">
            <h2 class="text-lg font-semibold mb-4">Test Payment</h2>
            
            <div class="mb-4">
                <label class="block text-sm font-medium text-gray-700 mb-1">
                    Test Phone Number *
                </label>
                <input type="text" id="testPhone" 
                       value="254708374149"
                       class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0A9961] focus:border-transparent"
                       placeholder="254708374149">
                <p class="text-xs text-gray-500 mt-1">
                    Use 254708374149 for sandbox testing
                </p>
            </div>
            
            <div class="mb-4">
                <label class="block text-sm font-medium text-gray-700 mb-1">
                    Amount (KES) *
                </label>
                <input type="number" id="testAmount" 
                       value="1" min="1" max="100"
                       class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0A9961] focus:border-transparent">
                <p class="text-xs text-gray-500 mt-1">
                    Use 1 KES for testing
                </p>
            </div>
            
            <button type="button" id="testButton"
                    class="w-full bg-[#0A9961] hover:bg-[#088753] text-white py-3 px-4 rounded-lg font-medium transition-colors duration-200">
                Test M-Pesa Payment
            </button>
            
            <div id="testResult" class="mt-4 hidden">
                <!-- Results will appear here -->
            </div>
        </div>
        
        <div class="mt-6 text-center">
            <a href="{{ url('/checkout') }}" class="text-[#0A9961] hover:underline">
                ← Back to Checkout
            </a>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const testButton = document.getElementById('testButton');
    const testResult = document.getElementById('testResult');
    
    testButton.addEventListener('click', async function() {
        const phone = document.getElementById('testPhone').value.trim();
        const amount = document.getElementById('testAmount').value;
        
        if (!phone || !amount) {
            alert('Please fill in all fields');
            return;
        }
        
        // Disable button
        testButton.disabled = true;
        testButton.innerHTML = 'Processing...';
        testResult.classList.add('hidden');
        
        try {
            const response = await fetch('{{ route("process.mpesa.payment") }}', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}',
                    'Accept': 'application/json'
                },
                body: JSON.stringify({
                    phone: phone,
                    amount: amount,
                    order_id: 'TEST_' + Date.now()
                })
            });
            
            const result = await response.json();
            
            // Show result
            testResult.classList.remove('hidden');
            
            if (result.success) {
                testResult.innerHTML = `
                    <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded">
                        <p class="font-bold">✓ Success!</p>
                        <p>${result.customer_message || 'Payment initiated successfully'}</p>
                        <p class="text-sm mt-2">Checkout Request ID: ${result.checkout_request_id}</p>
                        <p class="text-sm">Please check your phone for the payment prompt</p>
                        <div class="mt-3 pt-3 border-t border-green-300">
                            <button type="button" id="checkStatus" 
                                    class="text-sm bg-green-600 hover:bg-green-700 text-white px-3 py-1 rounded">
                                Check Payment Status
                            </button>
                        </div>
                    </div>
                `;
                
                // Add event listener to check status button
                setTimeout(() => {
                    const checkStatusBtn = document.getElementById('checkStatus');
                    if (checkStatusBtn) {
                        checkStatusBtn.addEventListener('click', function() {
                            checkPaymentStatus(result.checkout_request_id);
                        });
                    }
                }, 100);
                
                // Start auto-checking status
                setTimeout(() => checkPaymentStatus(result.checkout_request_id), 3000);
                
            } else {
                testResult.innerHTML = `
                    <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded">
                        <p class="font-bold">✗ Error:</p>
                        <p>${result.message}</p>
                    </div>
                `;
            }
            
        } catch (error) {
            testResult.classList.remove('hidden');
            testResult.innerHTML = `
                <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded">
                    <p class="font-bold">✗ Network Error:</p>
                    <p>${error.message}</p>
                </div>
            `;
        } finally {
            // Re-enable button
            testButton.disabled = false;
            testButton.innerHTML = 'Test M-Pesa Payment';
        }
    });
    
    async function checkPaymentStatus(checkoutRequestId) {
        const statusDiv = document.querySelector('#testResult div');
        if (!statusDiv) return;
        
        try {
            const response = await fetch(`{{ route('check.payment.status') }}?checkout_request_id=${checkoutRequestId}`);
            const result = await response.json();
            
            if (result.success && result.status === 'completed') {
                statusDiv.innerHTML += `
                    <div class="mt-2 p-2 bg-green-50 border border-green-200 rounded">
                        <p class="text-green-700 font-medium">✓ Payment Completed Successfully!</p>
                    </div>
                `;
            } else if (result.status === 'failed') {
                statusDiv.innerHTML += `
                    <div class="mt-2 p-2 bg-yellow-50 border border-yellow-200 rounded">
                        <p class="text-yellow-700">Payment Failed: ${result.message}</p>
                    </div>
                `;
            } else {
                // Check again after 3 seconds if still pending
                setTimeout(() => checkPaymentStatus(checkoutRequestId), 3000);
            }
        } catch (error) {
            console.error('Status check error:', error);
        }
    }
});
</script>
@endsection