@extends('layouts.app')

@section('title', $deal->title . ' - Finepoint Electronics')

@section('styles')
<style>
    /* Timer styles */
    .timer-box {
        background: linear-gradient(135deg, #0A9961 0%, #10B981 100%);
        color: white;
        border-radius: 8px;
    }
    
    /* Discount badge */
    .discount-badge {
        background: linear-gradient(135deg, #FF6B6B 0%, #FF8E53 100%);
        color: white;
    }
    
    /* Sticky add to cart */
    .sticky-cart {
        box-shadow: 0 -4px 20px rgba(0, 0, 0, 0.1);
    }
    
    /* Product card hover */
    .product-card:hover {
        transform: translateY(-4px);
        transition: transform 0.2s ease;
    }
    
    .line-clamp-2 {
        display: -webkit-box;
        -webkit-line-clamp: 2;
        -webkit-box-orient: vertical;
        overflow: hidden;
    }
    
    /* Timer animation */
    @keyframes pulse {
        0% { opacity: 1; }
        50% { opacity: 0.8; }
        100% { opacity: 1; }
    }
</style>
@endsection

@section('content')
    <!-- Breadcrumb -->
    <div class="bg-gray-100 border-b border-gray-200">
        <div class="max-w-7xl mx-auto px-4 py-3">
            <div class="flex items-center text-sm text-gray-600">
                <a href="{{ route('home') }}" class="hover:text-[#0A9961] no-underline">Home</a>
                <i class="fas fa-chevron-right mx-2 text-xs"></i>
                <a href="{{ route('deals.index') }}" class="hover:text-[#0A9961] no-underline">Deals</a>
                <i class="fas fa-chevron-right mx-2 text-xs"></i>
                <span class="text-gray-900 font-medium">{{ $deal->title }}</span>
            </div>
        </div>
    </div>

    <!-- Deal Expired Alert -->
    <div id="deal-expired-alert" class="hidden">
        <div class="bg-red-50 border-l-4 border-red-500 p-4">
            <div class="max-w-7xl mx-auto px-4">
                <div class="flex items-center">
                    <div class="flex-shrink-0">
                        <i class="fas fa-exclamation-triangle text-red-400 text-xl"></i>
                    </div>
                    <div class="ml-3">
                        <p class="text-red-800 font-medium">This deal has expired</p>
                        <p class="text-red-700 text-sm">The offer period for this deal has ended.</p>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Deal Header -->
    <div class="bg-white border-b border-gray-200">
        <div class="max-w-7xl mx-auto px-4 py-8">
            <div class="flex flex-col lg:flex-row lg:items-start lg:justify-between gap-8">
                <!-- Deal Image & Info -->
                <div class="lg:w-2/3">
                    <div class="flex flex-col md:flex-row gap-6">
                        <!-- Deal Image -->
                        <div class="md:w-1/3">
                            <div class="bg-gray-100 rounded-xl overflow-hidden shadow-lg">
                                @if($deal->image)
                                    <img src="{{ asset('storage/' . $deal->image) }}" 
                                         alt="{{ $deal->title }}"
                                         class="w-full h-64 object-cover">
                                @else
                                    <div class="w-full h-64 bg-gradient-to-br from-[#0A9961] to-[#10B981] flex items-center justify-center">
                                        <i class="fas fa-tag text-white text-4xl"></i>
                                    </div>
                                @endif
                            </div>
                        </div>
                        
                        <!-- Deal Details -->
                        <div class="md:w-2/3">
                            <div class="flex items-start justify-between mb-4">
                                <div>
                                    <h1 class="text-3xl font-bold text-gray-900 mb-2">{{ $deal->title }}</h1>
                                    <div class="flex items-center space-x-3 mb-4">
                                        @if($deal->featured)
                                        <span class="discount-badge text-xs font-bold px-3 py-1 rounded-full">
                                            <i class="fas fa-star mr-1"></i> FEATURED
                                        </span>
                                        @endif
                                        <span class="bg-[#0A9961] text-white text-xs font-bold px-3 py-1 rounded-full uppercase">
                                            {{ $deal->type }} DEAL
                                        </span>
                                    </div>
                                </div>
                                
                                <!-- Discount Badge -->
                                <div class="text-center">
                                    <div class="discount-badge rounded-xl p-4 min-w-24">
                                        <div class="text-2xl font-bold">
                                            @if($deal->type == 'percentage')
                                                {{ $deal->discount_value }}%
                                            @elseif($deal->type == 'fixed')
                                                Ksh {{ number_format($deal->discount_value) }}
                                            @else
                                                SPECIAL
                                            @endif
                                        </div>
                                        <div class="text-xs mt-1">OFF</div>
                                    </div>
                                </div>
                            </div>
                            
                            <p class="text-gray-700 mb-6">{{ $deal->description }}</p>
                            
                            <!-- Deal Stats -->
                            <div class="grid grid-cols-2 md:grid-cols-4 gap-4 mb-6">
                                <div class="bg-gray-50 rounded-lg p-3 text-center">
                                    <div class="text-2xl font-bold text-gray-900">{{ $deal->products->count() }}</div>
                                    <div class="text-xs text-gray-600">Products</div>
                                </div>
                                <div class="bg-gray-50 rounded-lg p-3 text-center">
                                    <div class="text-2xl font-bold text-gray-900">{{ $deal->used_count }}</div>
                                    <div class="text-xs text-gray-600">Times Used</div>
                                </div>
                                @if($deal->min_purchase)
                                <div class="bg-gray-50 rounded-lg p-3 text-center">
                                    <div class="text-2xl font-bold text-gray-900">Ksh {{ number_format($deal->min_purchase) }}</div>
                                    <div class="text-xs text-gray-600">Min. Purchase</div>
                                </div>
                                @endif
                                @if($deal->usage_limit)
                                <div class="bg-gray-50 rounded-lg p-3 text-center">
                                    <div class="text-2xl font-bold text-gray-900">{{ $deal->usage_limit }}</div>
                                    <div class="text-xs text-gray-600">Usage Limit</div>
                                </div>
                                @endif
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Timer Sidebar -->
                <div class="lg:w-1/3">
                    <div class="bg-gradient-to-br from-gray-900 to-gray-800 rounded-xl p-6 text-white" id="timer-container" 
                         data-end-date="{{ $deal->end_date->toISOString() }}"
                         data-start-date="{{ $deal->start_date->toISOString() }}">
                        <h3 class="text-xl font-bold mb-4 text-center">Offer Ends In</h3>
                        
                        <div id="timer" class="mb-6">
                            <div class="flex justify-center space-x-3">
                                <div class="text-center">
                                    <div class="text-3xl font-bold bg-gray-800 rounded-lg p-3 min-w-16 timer-digit" data-type="days">00</div>
                                    <div class="text-xs opacity-80 mt-2">Days</div>
                                </div>
                                <div class="text-3xl font-bold pt-3">:</div>
                                <div class="text-center">
                                    <div class="text-3xl font-bold bg-gray-800 rounded-lg p-3 min-w-16 timer-digit" data-type="hours">00</div>
                                    <div class="text-xs opacity-80 mt-2">Hours</div>
                                </div>
                                <div class="text-3xl font-bold pt-3">:</div>
                                <div class="text-center">
                                    <div class="text-3xl font-bold bg-gray-800 rounded-lg p-3 min-w-16 timer-digit" data-type="minutes">00</div>
                                    <div class="text-xs opacity-80 mt-2">Minutes</div>
                                </div>
                                <div class="text-3xl font-bold pt-3">:</div>
                                <div class="text-center">
                                    <div class="text-3xl font-bold bg-gray-800 rounded-lg p-3 min-w-16 timer-digit" data-type="seconds">00</div>
                                    <div class="text-xs opacity-80 mt-2">Seconds</div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="space-y-3">
                            <div class="flex justify-between text-sm">
                                <span class="opacity-80">Start Date:</span>
                                <span class="font-medium">{{ $deal->start_date->format('M d, Y h:i A') }}</span>
                            </div>
                            <div class="flex justify-between text-sm">
                                <span class="opacity-80">End Date:</span>
                                <span class="font-medium">{{ $deal->end_date->format('M d, Y h:i A') }}</span>
                            </div>
                            <div class="flex justify-between text-sm">
                                <span class="opacity-80">Status:</span>
                                <span class="font-medium {{ $deal->is_active ? 'text-green-400' : 'text-red-400' }}">
                                    {{ $deal->is_active ? 'Active' : 'Inactive' }}
                                </span>
                            </div>
                        </div>
                        
                        @auth
                            @if(auth()->user()->isAdmin())
                            <div class="mt-6 pt-6 border-t border-gray-700">
                                <a href="{{ route('admin.deals.edit', $deal) }}" 
                                   class="block w-full bg-blue-600 hover:bg-blue-700 text-white text-center py-2 rounded-lg font-medium no-underline transition duration-200">
                                    <i class="fas fa-edit mr-2"></i> Edit Deal
                                </a>
                            </div>
                            @endif
                        @endauth
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Products in Deal -->
    <div class="max-w-7xl mx-auto px-4 py-8">
        <div class="flex items-center justify-between mb-6">
            <h2 class="text-2xl font-bold text-gray-900">Products in This Deal</h2>
            <div class="text-sm text-gray-600">
                Showing {{ $deal->products->count() }} products
            </div>
        </div>
        
        @if($deal->products->count() > 0)
        <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
            @foreach($deal->products as $product)
            <div class="bg-white rounded-xl shadow-md overflow-hidden product-card border border-gray-200">
                <!-- Product Image -->
                <a href="{{ route('products.public.show', $product) }}" class="block no-underline">
                    <div class="relative h-48 bg-gray-50">
                        @if($product->image)
                            <img src="{{ $product->image_url }}" 
                                 alt="{{ $product->name }}"
                                 class="w-full h-full object-contain p-4">
                        @else
                            <div class="w-full h-full flex items-center justify-center">
                                <i class="fas fa-box text-gray-300 text-3xl"></i>
                            </div>
                        @endif
                        
                        <!-- Deal Badge -->
                        <div class="absolute top-3 right-3">
                            <div class="discount-badge text-xs font-bold px-3 py-1 rounded-full">
                                @if($deal->type == 'percentage')
                                    {{ $deal->discount_value }}% OFF
                                @elseif($deal->type == 'fixed')
                                    SAVE Ksh {{ number_format($deal->discount_value) }}
                                @else
                                    DEAL
                                @endif
                            </div>
                        </div>
                    </div>
                </a>
                
                <!-- Product Info -->
                <div class="p-4">
                    <h3 class="font-bold text-gray-900 mb-2">
                        <a href="{{ route('products.public.show', $product) }}" class="hover:text-[#0A9961] no-underline">
                            {{ $product->name }}
                        </a>
                    </h3>
                    
                    @if($product->brand)
                    <div class="mb-2">
                        <span class="text-xs text-gray-500">{{ $product->brand }}</span>
                    </div>
                    @endif
                    
                    <!-- Prices -->
                    <div class="mb-4">
                        @php
                            // Calculate deal price
                            $originalPrice = $product->price;
                            $dealPrice = $originalPrice;
                            
                            if ($deal->type == 'percentage') {
                                $discount = ($deal->discount_value / 100) * $originalPrice;
                                $dealPrice = $originalPrice - $discount;
                            } elseif ($deal->type == 'fixed') {
                                $dealPrice = max(0, $originalPrice - $deal->discount_value);
                            }
                        @endphp
                        
                        <div class="flex items-center space-x-2">
                            <span class="text-xl font-bold text-[#0A9961]">
                                Ksh {{ number_format($dealPrice) }}
                            </span>
                            <span class="text-sm text-gray-500 line-through">
                                Ksh {{ number_format($originalPrice) }}
                            </span>
                            @if($dealPrice < $originalPrice)
                            <span class="text-xs font-bold text-red-600">
                                Save Ksh {{ number_format($originalPrice - $dealPrice) }}
                            </span>
                            @endif
                        </div>
                    </div>
                    
                    <!-- Stock & Add to Cart -->
                    <div class="flex items-center justify-between">
                        <div>
                            @if($product->stock > 0)
                                @if($product->stock <= 10)
                                    <span class="text-xs text-yellow-600 bg-yellow-50 px-2 py-1 rounded">Low Stock</span>
                                @else
                                    <span class="text-xs text-green-600 bg-green-50 px-2 py-1 rounded">In Stock</span>
                                @endif
                            @else
                                <span class="text-xs text-red-600 bg-red-50 px-2 py-1 rounded">Out of Stock</span>
                            @endif
                        </div>
                        
                        <button onclick="addToCartWithDeal(event, {{ $product->id }}, '{{ addslashes($product->name) }}', {{ $dealPrice }}, {{ $product->stock }}, {{ $deal->id }})" 
                                class="bg-[#0A9961] hover:bg-[#088552] text-white p-2 rounded-lg {{ $product->stock <= 0 ? 'opacity-50 cursor-not-allowed' : '' }}"
                                {{ $product->stock <= 0 ? 'disabled' : '' }}>
                            <i class="fas fa-cart-plus"></i>
                        </button>
                    </div>
                </div>
            </div>
            @endforeach
        </div>
        @else
        <div class="text-center py-12 bg-white rounded-xl shadow-sm">
            <div class="w-20 h-20 rounded-full bg-gray-100 flex items-center justify-center mx-auto mb-4">
                <i class="fas fa-box-open text-gray-400 text-2xl"></i>
            </div>
            <h3 class="text-xl font-bold text-gray-900 mb-2">No Products in This Deal</h3>
            <p class="text-gray-600 mb-6">Check back later for products added to this deal.</p>
            <a href="{{ route('products.public') }}" 
               class="inline-block bg-[#0A9961] hover:bg-[#088552] text-white font-medium py-2 px-6 rounded-lg no-underline transition duration-200">
                Browse All Products
            </a>
        </div>
        @endif
    </div>

    <!-- Sticky Add All to Cart (for bundle deals) -->
    @if($deal->type == 'bundle' && $deal->products->count() > 0)
    <div class="sticky-cart fixed bottom-0 left-0 right-0 bg-white border-t border-gray-200 z-40">
        <div class="max-w-7xl mx-auto px-4 py-4">
            <div class="flex items-center justify-between">
                <div>
                    <div class="text-lg font-bold text-gray-900">Bundle Deal</div>
                    <div class="text-sm text-gray-600">
                        Get all {{ $deal->products->count() }} products at special price
                    </div>
                </div>
                
                <div class="flex items-center space-x-4">
                    <div class="text-right">
                        <div class="text-2xl font-bold text-[#0A9961]">
                            Ksh {{ number_format($deal->products->sum('discounted_price')) }}
                        </div>
                        <div class="text-sm text-gray-500 line-through">
                            Original: Ksh {{ number_format($deal->products->sum('price')) }}
                        </div>
                    </div>
                    
                    <button onclick="addBundleToCart({{ $deal->id }})" 
                            class="bg-gradient-to-r from-[#0A9961] to-[#10B981] hover:from-[#088552] hover:to-[#0EA371] text-white font-bold py-3 px-8 rounded-lg shadow-lg transition duration-200">
                        <i class="fas fa-shopping-cart mr-2"></i> Add Bundle to Cart
                    </button>
                </div>
            </div>
        </div>
    </div>
    @endif
@endsection

@section('scripts')
<script>
    // Add to cart function with deal support
    function addToCartWithDeal(event, productId, productName, dealPrice, stock, dealId) {
        event.preventDefault();
        event.stopPropagation();
        
        if (stock <= 0) {
            alert('This product is out of stock');
            return;
        }
        
        const button = event.target.closest('button');
        const originalHTML = button.innerHTML;
        button.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
        button.disabled = true;
        
        const csrfToken = document.querySelector('meta[name="csrf-token"]').getAttribute('content');
        const data = {
            product_id: productId,
            quantity: 1,
            deal_id: dealId, // Pass the deal ID
            _token: csrfToken
        };
        
        fetch('{{ route("cart.add") }}', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json',
                'X-CSRF-TOKEN': csrfToken
            },
            body: JSON.stringify(data)
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                let message = '✓ Added to cart: ' + productName;
                if (data.deal_applied) {
                    message += ' at deal price!';
                    if (data.original_price && data.deal_price) {
                        const savings = data.original_price - data.deal_price;
                        message += ` (Saved Ksh ${savings.toLocaleString()})`;
                    }
                }
                alert(message);
                updateCartCount(data.cart_count);
            } else {
                alert(data.message || 'Failed to add to cart');
            }
        })
        .catch(error => {
            alert('An error occurred. Please try again.');
        })
        .finally(() => {
            button.innerHTML = originalHTML;
            button.disabled = false;
        });
    }
    
    // Original add to cart function (for reference)
    function addToCart(event, productId, productName, price, stock) {
        event.preventDefault();
        event.stopPropagation();
        
        if (stock <= 0) {
            alert('This product is out of stock');
            return;
        }
        
        const button = event.target.closest('button');
        const originalHTML = button.innerHTML;
        button.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
        button.disabled = true;
        
        const csrfToken = document.querySelector('meta[name="csrf-token"]').getAttribute('content');
        const data = {
            product_id: productId,
            quantity: 1,
            _token: csrfToken
        };
        
        fetch('{{ route("cart.add") }}', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json',
                'X-CSRF-TOKEN': csrfToken
            },
            body: JSON.stringify(data)
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert('✓ Added to cart: ' + productName);
                updateCartCount(data.cart_count);
            } else {
                alert(data.message || 'Failed to add to cart');
            }
        })
        .catch(error => {
            alert('An error occurred. Please try again.');
        })
        .finally(() => {
            button.innerHTML = originalHTML;
            button.disabled = false;
        });
    }
    
    // Update bundle add to cart function
    function addBundleToCart(dealId) {
        const csrfToken = document.querySelector('meta[name="csrf-token"]').getAttribute('content');
        const productIds = @json($deal->products->pluck('id'));
        
        const button = event.target;
        const originalHTML = button.innerHTML;
        button.innerHTML = '<i class="fas fa-spinner fa-spin mr-2"></i> Adding Bundle...';
        button.disabled = true;
        
        let completed = 0;
        const totalProducts = productIds.length;
        let successCount = 0;
        
        productIds.forEach(productId => {
            const data = {
                product_id: productId,
                quantity: 1,
                deal_id: dealId, // Pass deal ID for bundle
                _token: csrfToken
            };
            
            fetch('{{ route("cart.add") }}', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Accept': 'application/json',
                    'X-CSRF-TOKEN': csrfToken
                },
                body: JSON.stringify(data)
            })
            .then(response => response.json())
            .then(data => {
                completed++;
                if (data.success) {
                    successCount++;
                }
                
                if (completed === totalProducts) {
                    if (successCount === totalProducts) {
                        alert('✓ Added entire bundle to cart with deal prices!');
                    } else if (successCount > 0) {
                        alert(`✓ Added ${successCount} out of ${totalProducts} bundle items to cart.`);
                    } else {
                        alert('Failed to add bundle to cart. Please try again.');
                    }
                    
                    updateCartCount();
                    button.innerHTML = originalHTML;
                    button.disabled = false;
                }
            })
            .catch(error => {
                completed++;
                console.error('Error adding bundle item:', error);
                
                if (completed === totalProducts) {
                    alert('An error occurred while adding the bundle. Please try again.');
                    button.innerHTML = originalHTML;
                    button.disabled = false;
                }
            });
        });
    }
    
    function updateCartCount(count = null) {
        if (count !== null) {
            setCartCount(count);
        } else {
            fetch('{{ route("cart.count") }}')
                .then(response => response.json())
                .then(data => {
                    setCartCount(data.cart_count);
                });
        }
    }
    
    function setCartCount(count) {
        const cartCountElement = document.getElementById('cart-count');
        if (cartCountElement) {
            cartCountElement.textContent = count;
        }
    }
    
    // Timer function
    function startDealTimer() {
        const timerContainer = document.getElementById('timer-container');
        if (!timerContainer) return;
        
        const endDateStr = timerContainer.dataset.endDate;
        const endDate = new Date(endDateStr);
        
        if (isNaN(endDate.getTime())) return;
        
        updateTimer(endDate);
        setInterval(() => updateTimer(endDate), 1000);
    }
    
    function updateTimer(endDate) {
        const now = new Date();
        const timeLeft = endDate - now;
        
        if (timeLeft <= 0) {
            document.getElementById('timer').innerHTML = `
                <div class="text-center">
                    <div class="text-2xl font-bold text-red-400 mb-2">DEAL EXPIRED</div>
                    <div class="text-sm opacity-80">This offer has ended</div>
                </div>
            `;
            document.getElementById('deal-expired-alert').classList.remove('hidden');
            return;
        }
        
        const days = Math.floor(timeLeft / (1000 * 60 * 60 * 24));
        const hours = Math.floor((timeLeft % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
        const minutes = Math.floor((timeLeft % (1000 * 60 * 60)) / (1000 * 60));
        const seconds = Math.floor((timeLeft % (1000 * 60)) / 1000);
        
        const daysElement = document.querySelector('[data-type="days"]');
        const hoursElement = document.querySelector('[data-type="hours"]');
        const minutesElement = document.querySelector('[data-type="minutes"]');
        const secondsElement = document.querySelector('[data-type="seconds"]');
        
        if (daysElement) daysElement.textContent = days.toString().padStart(2, '0');
        if (hoursElement) hoursElement.textContent = hours.toString().padStart(2, '0');
        if (minutesElement) minutesElement.textContent = minutes.toString().padStart(2, '0');
        if (secondsElement) secondsElement.textContent = seconds.toString().padStart(2, '0');
        
        if (secondsElement) {
            secondsElement.style.animation = 'none';
            setTimeout(() => {
                secondsElement.style.animation = 'pulse 1s ease';
            }, 10);
        }
    }
    
    @if($deal->type == 'bundle')
    document.addEventListener('DOMContentLoaded', function() {
        const stickyCart = document.querySelector('.sticky-cart');
        if (stickyCart) {
            document.body.style.paddingBottom = stickyCart.offsetHeight + 'px';
        }
    });
    @endif
    
    document.addEventListener('DOMContentLoaded', function() {
        updateCartCount();
        startDealTimer();
    });
</script>
@endsection