@extends('layouts.app')

@section('title', 'Hot Deals - Finepoint Electronics')

@section('styles')
<style>
    @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');
    
    .deal-card { 
        transition: all 0.2s ease-in-out; 
    }
    
    .deal-card:hover { 
        transform: translateY(-2px); 
        box-shadow: 0 10px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04); 
    }
    
    .line-clamp-2 { 
        display: -webkit-box; 
        -webkit-line-clamp: 2; 
        -webkit-box-orient: vertical; 
        overflow: hidden; 
    }
    
    /* Timer styles */
    .timer-box {
        background: linear-gradient(135deg, #0A9961 0%, #10B981 100%);
        color: white;
        border-radius: 8px;
    }
    
    /* Featured deal badge */
    .featured-badge {
        background: linear-gradient(135deg, #FF6B6B 0%, #FF8E53 100%);
        animation: pulse 2s infinite;
    }
    
    @keyframes pulse {
        0% { transform: scale(1); }
        50% { transform: scale(1.05); }
        100% { transform: scale(1); }
    }
</style>
@endsection

@section('content')
    <!-- Hero Section for Deals -->
    <div class="bg-gradient-to-r from-[#0A9961] to-[#10B981] text-white">
        <div class="max-w-7xl mx-auto px-4 py-2">
            <div class="text-center">
                <h1 class="text-4xl font-bold mb-4">🔥 Hot Deals</h1>
                <p class="text-gray-100 text-lg mb-6">Limited time offers on premium electronics</p>
                
                @if($featuredDeals->count() > 0)
                <div class="mt-8 grid grid-cols-1 md:grid-cols-3 gap-6">
                    @foreach($featuredDeals as $deal)
                    <a href="{{ route('deals.show', $deal->slug) }}" class="block no-underline">
                        <div class="bg-white/10 backdrop-blur-sm rounded-xl p-6 hover:bg-white/20 transition-all duration-300">
                            <div class="flex items-center justify-between mb-4">
                                <span class="featured-badge text-xs font-bold px-3 py-1 rounded-full">FEATURED</span>
                                <span class="text-sm bg-red-500 px-3 py-1 rounded-full">
                                    <i class="fas fa-clock mr-1"></i> Ends {{ $deal->end_date->diffForHumans() }}
                                </span>
                            </div>
                            <h3 class="text-xl font-bold mb-2">{{ $deal->title }}</h3>
                            <p class="text-gray-200 text-sm mb-4 line-clamp-2">{{ $deal->description }}</p>
                            <div class="flex items-center">
                                <span class="text-2xl font-bold mr-2">
                                    @if($deal->type == 'percentage')
                                        {{ $deal->discount_value }}% OFF
                                    @elseif($deal->type == 'fixed')
                                        Save Ksh {{ number_format($deal->discount_value) }}
                                    @else
                                        Special Offer
                                    @endif
                                </span>
                                <span class="ml-auto text-sm">
                                    {{ $deal->products_count ?? $deal->products->count() }} products
                                </span>
                            </div>
                        </div>
                    </a>
                    @endforeach
                </div>
                @endif
            </div>
        </div>
    </div>

    <!-- All Deals Section -->
    <div class="max-w-7xl mx-auto px-4 py-8">
        <div class="flex justify-between items-center mb-8">
            <h2 class="text-2xl font-bold text-gray-900">All Active Deals</h2>
            <div class="text-sm text-gray-600">
                <i class="fas fa-fire text-red-500 mr-2"></i>
                {{ $deals->count() }} deals available
            </div>
        </div>

        @if($deals->count() > 0)
        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
            @foreach($deals as $deal)
            <div class="bg-white rounded-xl shadow-md overflow-hidden deal-card" 
                 data-deal-id="{{ $deal->id }}" 
                 data-end-date="{{ $deal->end_date->toISOString() }}">
                <!-- Deal Header -->
                <div class="p-6 border-b border-gray-100">
                    <div class="flex items-start justify-between mb-4">
                        <div>
                            <h3 class="text-xl font-bold text-gray-900 mb-2">{{ $deal->title }}</h3>
                            <p class="text-gray-600 text-sm line-clamp-2">{{ $deal->description }}</p>
                        </div>
                        @if($deal->featured)
                        <span class="featured-badge text-xs font-bold px-3 py-1 rounded-full">FEATURED</span>
                        @endif
                    </div>
                    
                    <!-- Deal Info -->
                    <div class="flex items-center justify-between mb-4">
                        <div class="flex items-center space-x-4">
                            <div class="text-center">
                                <div class="text-2xl font-bold text-[#0A9961]">
                                    @if($deal->type == 'percentage')
                                        {{ $deal->discount_value }}% OFF
                                    @elseif($deal->type == 'fixed')
                                        Ksh {{ number_format($deal->discount_value) }} OFF
                                    @else
                                        <span class="text-lg">Special Bundle</span>
                                    @endif
                                </div>
                                <div class="text-xs text-gray-500 mt-1">{{ strtoupper($deal->type) }} DEAL</div>
                            </div>
                            
                            @if($deal->min_purchase)
                            <div class="text-center">
                                <div class="text-sm font-semibold text-gray-700">Min. Purchase</div>
                                <div class="text-sm text-gray-600">Ksh {{ number_format($deal->min_purchase) }}</div>
                            </div>
                            @endif
                        </div>
                        
                        <a href="{{ route('deals.show', $deal->slug) }}" 
                           class="bg-[#0A9961] hover:bg-[#088552] text-white px-4 py-2 rounded-lg text-sm font-medium no-underline transition duration-200">
                            View Deal
                        </a>
                    </div>
                    
                    <!-- Timer -->
                    <div class="timer-box p-3 text-center">
                        <div class="text-sm font-medium mb-1">Offer ends in:</div>
                        <div class="flex justify-center space-x-2 timer-display" data-deal-id="{{ $deal->id }}">
                            <div class="text-center">
                                <div class="text-xl font-bold timer-days" data-deal-id="{{ $deal->id }}">00</div>
                                <div class="text-xs opacity-80">Days</div>
                            </div>
                            <div class="text-2xl font-bold">:</div>
                            <div class="text-center">
                                <div class="text-xl font-bold timer-hours" data-deal-id="{{ $deal->id }}">00</div>
                                <div class="text-xs opacity-80">Hours</div>
                            </div>
                            <div class="text-2xl font-bold">:</div>
                            <div class="text-center">
                                <div class="text-xl font-bold timer-minutes" data-deal-id="{{ $deal->id }}">00</div>
                                <div class="text-xs opacity-80">Mins</div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Deal Products Preview -->
                <div class="p-6">
                    <h4 class="font-medium text-gray-900 mb-3">Products in this deal:</h4>
                    @if($deal->products->count() > 0)
                    <div class="grid grid-cols-2 gap-3">
                        @foreach($deal->products->take(4) as $product)
                        <a href="{{ route('products.public.show', $product) }}" class="no-underline">
                            <div class="flex items-center space-x-3 p-2 hover:bg-gray-50 rounded-lg">
                                @if($product->image)
                                <img src="{{ $product->image_url }}" 
                                     alt="{{ $product->name }}"
                                     class="w-12 h-12 rounded-lg object-cover">
                                @else
                                <div class="w-12 h-12 rounded-lg bg-gray-100 flex items-center justify-center">
                                    <i class="fas fa-box text-gray-400"></i>
                                </div>
                                @endif
                                <div class="flex-1 min-w-0">
                                    <div class="text-sm font-medium text-gray-900 truncate">{{ $product->name }}</div>
                                    <div class="flex items-center">
                                        @if($product->is_on_sale)
                                        <span class="text-xs font-bold text-[#0A9961] mr-2">
                                            Ksh {{ number_format($product->discounted_price) }}
                                        </span>
                                        <span class="text-xs text-gray-500 line-through">
                                            Ksh {{ number_format($product->price) }}
                                        </span>
                                        @else
                                        <span class="text-xs text-gray-900">
                                            Ksh {{ number_format($product->price) }}
                                        </span>
                                        @endif
                                    </div>
                                </div>
                            </div>
                        </a>
                        @endforeach
                        
                        @if($deal->products->count() > 4)
                        <div class="col-span-2 text-center">
                            <span class="text-sm text-[#0A9961] font-medium">
                                +{{ $deal->products->count() - 4 }} more products
                            </span>
                        </div>
                        @endif
                    </div>
                    @else
                    <div class="text-center py-4 text-gray-500">
                        <i class="fas fa-box-open text-2xl mb-2"></i>
                        <p>No products in this deal yet</p>
                    </div>
                    @endif
                </div>
                
                <!-- Deal Footer -->
                <div class="bg-gray-50 px-6 py-3 border-t border-gray-100">
                    <div class="flex items-center justify-between text-sm text-gray-600">
                        <div class="flex items-center">
                            <i class="fas fa-users mr-2"></i>
                            <span>Used {{ $deal->used_count }} times</span>
                            @if($deal->usage_limit)
                            <span class="mx-2">•</span>
                            <span>Limit: {{ $deal->usage_limit }}</span>
                            @endif
                        </div>
                        <div class="flex items-center">
                            <i class="far fa-calendar-alt mr-2"></i>
                            <span>Ends {{ $deal->end_date->format('M d, Y') }}</span>
                        </div>
                    </div>
                </div>
            </div>
            @endforeach
        </div>
        @else
        <!-- No Deals Available -->
        <div class="text-center py-16 bg-white rounded-xl shadow-sm">
            <div class="w-24 h-24 rounded-full bg-gray-100 flex items-center justify-center mx-auto mb-6">
                <i class="fas fa-tag text-gray-400 text-3xl"></i>
            </div>
            <h3 class="text-xl font-bold text-gray-900 mb-2">No Active Deals</h3>
            <p class="text-gray-600 mb-6">Check back later for amazing offers!</p>
            <a href="{{ route('products.public') }}" 
               class="inline-block bg-[#0A9961] hover:bg-[#088552] text-white font-medium py-2 px-6 rounded-lg no-underline transition duration-200">
                Browse Products
            </a>
        </div>
        @endif
    </div>
@endsection

@section('scripts')
<script>
    // Timer function for each deal
    function updateTimer(dealId, endDate) {
        const end = new Date(endDate).getTime();
        const now = new Date().getTime();
        const distance = end - now;
        
        if (distance < 0) {
            const timerDisplay = document.querySelector(`.timer-display[data-deal-id="${dealId}"]`);
            if (timerDisplay) {
                timerDisplay.innerHTML = "<div class='text-center w-full'><span class='text-sm font-medium'>EXPIRED</span></div>";
            }
            return;
        }
        
        const days = Math.floor(distance / (1000 * 60 * 60 * 24));
        const hours = Math.floor((distance % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
        const minutes = Math.floor((distance % (1000 * 60 * 60)) / (1000 * 60));
        
        // Update display elements
        const daysElement = document.querySelector(`.timer-days[data-deal-id="${dealId}"]`);
        const hoursElement = document.querySelector(`.timer-hours[data-deal-id="${dealId}"]`);
        const minutesElement = document.querySelector(`.timer-minutes[data-deal-id="${dealId}"]`);
        
        if (daysElement) daysElement.textContent = days.toString().padStart(2, '0');
        if (hoursElement) hoursElement.textContent = hours.toString().padStart(2, '0');
        if (minutesElement) minutesElement.textContent = minutes.toString().padStart(2, '0');
    }
    
    // Initialize all timers
    function initializeTimers() {
        const dealCards = document.querySelectorAll('[data-deal-id]');
        
        dealCards.forEach(card => {
            const dealId = card.getAttribute('data-deal-id');
            const endDate = card.getAttribute('data-end-date');
            
            // Update immediately
            updateTimer(dealId, endDate);
            
            // Update every minute (60000ms)
            setInterval(() => {
                updateTimer(dealId, endDate);
            }, 60000);
        });
    }
    
    // Initialize when DOM is loaded
    document.addEventListener('DOMContentLoaded', function() {
        initializeTimers();
    });
</script>
@endsection