@extends('layouts.app')

@section('title', 'Shopping Cart - Finepoint Electronics')

@section('content')
<div class="max-w-7xl mx-auto px-4 py-8">
    <h1 class="text-2xl font-bold text-gray-900 mb-6">Shopping Cart</h1>
    
    @if(($cartItems ?? collect())->count() > 0)
    <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
        <!-- Cart Items -->
        <div class="lg:col-span-2">
            <div class="bg-white rounded-lg shadow-sm border border-gray-200">
                <!-- Cart Header -->
                <div class="px-6 py-4 border-b border-gray-200 flex justify-between items-center">
                    <h2 class="font-medium text-gray-900">Your Items ({{ $cartItems->count() }})</h2>
                    <button onclick="clearCart()" class="text-red-600 hover:text-red-800 text-sm">
                        <i class="fas fa-trash-alt mr-1"></i> Clear Cart
                    </button>
                </div>
                
                <!-- Discount Alert -->
                @if($totalDiscount > 0)
                <div class="px-6 py-3 bg-green-50 border-b border-green-100">
                    <div class="flex items-center">
                        <i class="fas fa-tag text-green-600 mr-2"></i>
                        <span class="text-green-800 font-medium">You're saving Ksh {{ number_format($totalDiscount) }} with deals!</span>
                    </div>
                </div>
                @endif
                
                <!-- Cart Items List -->
                <div class="divide-y divide-gray-100">
                    @foreach($cartItems as $cartItem)
                    @php
                        $product = $cartItem->product;
                        
                        // Determine cart item ID based on auth status
                        if (Auth::check()) {
                            $cartItemId = $cartItem->id;
                            $cartKey = $cartItemId;
                        } else {
                            $cartItemId = $cartItem->cart_key ?? $product->id;
                            $cartKey = $cartItemId;
                        }
                        
                        // Get prices
                        $originalPrice = $cartItem->original_price ?? $product->price;
                        $actualPrice = $cartItem->actual_price ?? $product->price;
                        $quantity = $cartItem->quantity;
                        $hasDeal = $cartItem->has_deal ?? (!empty($cartItem->deal_id));
                        
                        // Calculate totals
                        $itemOriginalTotal = $originalPrice * $quantity;
                        $itemActualTotal = $actualPrice * $quantity;
                        $itemDiscount = $itemOriginalTotal - $itemActualTotal;
                    @endphp
                    
                    <div id="cart-row-{{ $cartKey }}" class="px-6 py-4 flex items-center" 
                         data-price="{{ $actualPrice }}" 
                         data-original-price="{{ $originalPrice }}"
                         data-product-id="{{ $product->id }}"
                         data-has-deal="{{ $hasDeal ? '1' : '0' }}">
                        
                        <!-- Product Image -->
                        <div class="w-20 h-20 flex-shrink-0">
                            @if($product->image)
                                <img src="{{ $product->image_url }}" 
                                     alt="{{ $product->name }}"
                                     class="w-full h-full object-contain">
                            @else
                                <div class="w-full h-full bg-gray-100 rounded flex items-center justify-center">
                                    <i class="fas fa-box text-gray-300"></i>
                                </div>
                            @endif
                        </div>
                        
                        <!-- Product Info -->
                        <div class="ml-4 flex-grow">
                            <a href="{{ route('products.public.show', $product) }}" 
                               class="font-medium text-gray-900 hover:text-[#0A9961] block">
                                {{ $product->name }}
                            </a>
                            
                            <!-- Price Display -->
                            <div class="mt-1">
                                @if($hasDeal && $originalPrice > $actualPrice)
                                    <div class="flex items-center">
                                        <span class="text-[#0A9961] font-bold">
                                            Ksh {{ number_format($actualPrice) }}
                                        </span>
                                        <span class="text-gray-500 text-sm line-through ml-2">
                                            Ksh {{ number_format($originalPrice) }}
                                        </span>
                                        <span class="ml-2 bg-red-100 text-red-800 text-xs font-medium px-2 py-0.5 rounded">
                                            @if($cartItem->deal_type == 'percentage')
                                                {{ $cartItem->deal_discount_value }}% OFF
                                            @else
                                                SAVE Ksh {{ number_format($itemDiscount / $quantity) }}
                                            @endif
                                        </span>
                                    </div>
                                @else
                                    <span class="text-[#0A9961] font-bold">
                                        Ksh {{ number_format($actualPrice) }}
                                    </span>
                                @endif
                            </div>
                            
                            <!-- Deal Info -->
                            @if($hasDeal && isset($cartItem->deal_type))
                                <div class="text-xs text-gray-600 mt-1">
                                    <i class="fas fa-bolt text-yellow-500 mr-1"></i>
                                    {{ $cartItem->deal_type == 'percentage' ? $cartItem->deal_discount_value . '% Discount' : 'Fixed Discount' }}
                                </div>
                            @endif
                            
                            <!-- Stock Warning -->
                            @if($product->stock < $quantity)
                                <p class="text-red-600 text-xs mt-1">
                                    <i class="fas fa-exclamation-triangle"></i> 
                                    Only {{ $product->stock }} in stock
                                </p>
                            @endif
                        </div>
                        
                        <!-- Quantity Controls -->
                        <div class="flex items-center">
                            <button type="button" onclick="updateQuantity('{{ $cartKey }}', -1)" 
                                    class="w-8 h-8 flex items-center justify-center border border-gray-300 rounded-l hover:bg-gray-50">
                                <i class="fas fa-minus text-gray-600 text-xs"></i>
                            </button>
                            <input type="number" 
                                   id="quantity-input-{{ $cartKey }}"
                                   value="{{ $quantity }}"
                                   min="1"
                                   max="{{ $product->stock }}"
                                   class="w-12 h-8 text-center border-t border-b border-gray-300 focus:outline-none focus:border-[#0A9961]">
                            <button type="button" onclick="updateQuantity('{{ $cartKey }}', 1)" 
                                    class="w-8 h-8 flex items-center justify-center border border-gray-300 rounded-r hover:bg-gray-50">
                                <i class="fas fa-plus text-gray-600 text-xs"></i>
                            </button>
                        </div>
                        
                        <!-- Item Total -->
                        <div class="ml-6 text-right">
                            @if($hasDeal && $itemDiscount > 0)
                                <div class="mb-1">
                                    <span id="item-total-{{ $cartKey }}" class="font-bold text-gray-900">
                                        Ksh {{ number_format($itemActualTotal) }}
                                    </span>
                                    <div class="text-xs text-gray-500 line-through">
                                        Ksh {{ number_format($itemOriginalTotal) }}
                                    </div>
                                </div>
                            @else
                                <p id="item-total-{{ $cartKey }}" class="font-bold text-gray-900">
                                    Ksh {{ number_format($itemActualTotal) }}
                                </p>
                            @endif
                            
                            <button type="button" onclick="removeFromCart('{{ $cartKey }}', '{{ addslashes($product->name) }}')" 
                                    class="text-red-600 hover:text-red-800 text-sm mt-1">
                                <i class="fas fa-trash-alt mr-1"></i> Remove
                            </button>
                        </div>
                    </div>
                    @endforeach
                </div>
                
                <!-- Cart Summary (Mobile) -->
                <div class="lg:hidden border-t border-gray-200 p-6">
                    <div class="space-y-3">
                        @if($totalDiscount > 0)
                        <div class="flex justify-between">
                            <span class="text-gray-600">Original Subtotal</span>
                            <span class="line-through text-gray-500">Ksh {{ number_format($originalSubtotal ?? $subtotal) }}</span>
                        </div>
                        <div class="flex justify-between">
                            <span class="text-gray-600">Discount</span>
                            <span class="text-green-600 font-medium">- Ksh {{ number_format($totalDiscount) }}</span>
                        </div>
                        @endif
                        <div class="flex justify-between">
                            <span class="text-gray-600">Subtotal</span>
                            <span class="font-medium">Ksh {{ number_format($subtotal) }}</span>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Order Summary -->
        <div class="lg:col-span-1">
            <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 sticky top-24">
                <h2 class="font-bold text-gray-900 text-lg mb-4">Order Summary</h2>
                
                <div class="space-y-3 mb-6">
                    @if($totalDiscount > 0)
                    <div class="flex justify-between">
                        <span class="text-gray-600">Original Subtotal</span>
                        <span class="line-through text-gray-500">Ksh {{ number_format($originalSubtotal ?? $subtotal) }}</span>
                    </div>
                    <div class="flex justify-between">
                        <span class="text-gray-600">Discount</span>
                        <span class="text-green-600 font-medium">- Ksh {{ number_format($totalDiscount) }}</span>
                    </div>
                    @endif
                    
                    <div class="flex justify-between">
                        <span class="text-gray-600">Subtotal</span>
                        <span id="subtotal" class="font-medium">Ksh {{ number_format($subtotal) }}</span>
                    </div>
                    
                    <div class="flex justify-between">
                        <span class="text-gray-600">Shipping</span>
                        <span class="font-medium text-[#0A9961]">Calculated at checkout</span>
                    </div>
                    
                    <div class="border-t border-gray-200 pt-3">
                        <div class="flex justify-between">
                            <span class="font-bold text-gray-900">Total</span>
                            <span id="total" class="font-bold text-[#0A9961] text-xl">Ksh {{ number_format($subtotal) }}</span>
                        </div>
                    </div>
                    
                    @if($totalDiscount > 0)
                    <div class="bg-green-50 border border-green-200 rounded-lg p-3 mt-2">
                        <div class="flex items-center">
                            <i class="fas fa-tag text-green-600 mr-2"></i>
                            <div>
                                <p class="text-green-800 text-sm font-medium">Great deal!</p>
                                <p class="text-green-700 text-xs">You saved Ksh {{ number_format($totalDiscount) }}</p>
                            </div>
                        </div>
                    </div>
                    @endif
                </div>
                
                @if($cartItems->count() > 0)
                    <div class="mt-6">
                        @if(auth()->check())
                            <a href="{{ route('checkout') }}" 
                            class="w-full bg-[#0A9961] hover:bg-[#088753] text-white py-3 px-4 rounded-lg font-medium text-center transition-colors duration-200 block">
                                Proceed to Checkout
                            </a>
                        @else
                            <button onclick="redirectToLogin()" 
                                    class="w-full bg-[#0A9961] hover:bg-[#088753] text-white py-3 px-4 rounded-lg font-medium text-center transition-colors duration-200 block">
                                Login to Checkout
                            </button>
                        @endif
                    </div>
                @endif
                
                <!-- Continue Shopping -->
                <div class="mt-6">
                    <a href="{{ route('products.public') }}" 
                       class="w-full border border-[#0A9961] text-[#0A9961] hover:bg-[#0A9961] hover:text-white py-2 px-4 rounded-lg font-medium text-center transition-colors duration-200 block">
                        Continue Shopping
                    </a>
                </div>
                
                <!-- Security Info -->
                <div class="mt-6 pt-6 border-t border-gray-200">
                    <div class="flex items-center text-xs text-gray-500">
                        <i class="fas fa-lock mr-2"></i>
                        <span>Secure checkout · Your payment information is encrypted</span>
                    </div>
                </div>
            </div>
        </div>
    </div>
    @else
    <!-- Empty Cart -->
    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-12 text-center">
        <div class="w-16 h-16 rounded-full bg-gray-100 flex items-center justify-center mx-auto mb-4">
            <i class="fas fa-shopping-cart text-gray-400 text-xl"></i>
        </div>
        <h2 class="text-xl font-bold text-gray-900 mb-2">Your cart is empty</h2>
        <p class="text-gray-600 mb-6">Add some products to get started!</p>
        <a href="{{ route('products.public') }}" 
           class="inline-block bg-[#0A9961] hover:bg-[#088552] text-white font-medium py-2 px-6 rounded transition duration-200">
            Browse Products
        </a>
    </div>
    @endif
</div>
@endsection

@push('scripts')
<script>
    // Simple notification function
    function showNotification(message, type = 'success') {
        // Create a simple toast notification
        const toast = document.createElement('div');
        toast.className = `fixed top-4 right-4 px-4 py-2 rounded-md text-white ${
            type === 'success' ? 'bg-green-500' : 'bg-red-500'
        }`;
        toast.textContent = message;
        document.body.appendChild(toast);
        
        setTimeout(() => {
            toast.remove();
        }, 3000);
    }
    
    // FIXED: Calculate and update cart totals WITHOUT recursion
    function updateCartTotals() {
        let subtotal = 0;
        let originalSubtotal = 0;
        
        // Get all cart items and calculate totals
        document.querySelectorAll('[id^="cart-row-"]').forEach(row => {
            const cartKey = row.id.replace('cart-row-', '');
            const input = document.getElementById(`quantity-input-${cartKey}`);
            
            if (input) {
                const quantity = parseInt(input.value) || 0;
                const price = parseFloat(row.getAttribute('data-price')) || 0;
                const originalPrice = parseFloat(row.getAttribute('data-original-price')) || price;
                
                subtotal += price * quantity;
                originalSubtotal += originalPrice * quantity;
            }
        });
        
        // Update display
        const subtotalElement = document.getElementById('subtotal');
        const totalElement = document.getElementById('total');
        
        if (subtotalElement) {
            subtotalElement.textContent = `Ksh ${subtotal.toLocaleString()}`;
        }
        
        if (totalElement) {
            totalElement.textContent = `Ksh ${subtotal.toLocaleString()}`;
        }
    }
    
    // FIXED: Update single item total display
    function updateSingleItemTotal(cartKey) {
        const row = document.getElementById(`cart-row-${cartKey}`);
        if (!row) return;
        
        const input = document.getElementById(`quantity-input-${cartKey}`);
        if (!input) return;
        
        const quantity = parseInt(input.value) || 0;
        const price = parseFloat(row.getAttribute('data-price')) || 0;
        const originalPrice = parseFloat(row.getAttribute('data-original-price')) || price;
        const hasDeal = row.getAttribute('data-has-deal') === '1';
        
        const itemActualTotal = price * quantity;
        const itemOriginalTotal = originalPrice * quantity;
        
        const itemTotalElement = document.getElementById(`item-total-${cartKey}`);
        
        if (itemTotalElement) {
            if (hasDeal && itemOriginalTotal > itemActualTotal) {
                itemTotalElement.innerHTML = `
                    <span class="font-bold text-gray-900">Ksh ${itemActualTotal.toLocaleString()}</span>
                    <div class="text-xs text-gray-500 line-through">Ksh ${itemOriginalTotal.toLocaleString()}</div>
                `;
            } else {
                itemTotalElement.textContent = `Ksh ${itemActualTotal.toLocaleString()}`;
            }
        }
        
        // Return the calculated totals for this item
        return {
            subtotal: itemActualTotal,
            originalSubtotal: itemOriginalTotal
        };
    }
    
    // FIXED: Update quantity function - calls updateCartTotals separately
    function updateQuantity(cartKey, change) {
        const input = document.getElementById(`quantity-input-${cartKey}`);
        if (!input) return;
        
        let newQuantity = parseInt(input.value) + change;
        
        if (newQuantity < 1) {
            newQuantity = 1;
        }
        
        // Check max stock
        const maxStock = parseInt(input.getAttribute('max')) || 999;
        if (newQuantity > maxStock) {
            newQuantity = maxStock;
            showNotification(`Maximum ${maxStock} items available`, 'error');
            return;
        }
        
        input.value = newQuantity;
        
        // Update this item's total display
        updateSingleItemTotal(cartKey);
        
        // Update cart totals separately
        updateCartTotals();
        
        // Send to server
        saveQuantityToServer(cartKey, newQuantity);
    }
    
    // Save quantity to server
    function saveQuantityToServer(cartKey, quantity) {
        const csrfToken = document.querySelector('meta[name="csrf-token"]').getAttribute('content');
        
        // Use FormData for better compatibility
        const formData = new FormData();
        formData.append('_token', csrfToken);
        formData.append('quantity', quantity);
        
        // Determine if we're updating by cart item ID or cart key
        const isCartItemId = cartKey.toString().match(/^\d+$/); // Check if it's numeric ID
        
        let url = isCartItemId 
            ? `/cart/update/${cartKey}` 
            : `/cart/update/${cartKey}`; // For guests, cartKey is the productId_dealId
        
        fetch(url, {
            method: 'POST',
            body: formData
        })
        .then(response => {
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            return response.json();
        })
        .then(data => {
            if (data.success) {
                // Update cart count in navbar
                updateCartCount();
                showNotification('Cart updated successfully');
            } else {
                showNotification(data.message || 'Failed to update quantity', 'error');
                // Reload to get correct values
                window.location.reload();
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showNotification('An error occurred. Please try again.', 'error');
            window.location.reload();
        });
    }
    
    // Remove item from cart
    function removeFromCart(cartKey, productName) {
        if (!confirm(`Remove "${productName}" from cart?`)) {
            return;
        }
        
        const csrfToken = document.querySelector('meta[name="csrf-token"]').getAttribute('content');
        
        // Use FormData
        const formData = new FormData();
        formData.append('_token', csrfToken);
        
        // Determine if we're removing by cart item ID or cart key
        const isCartItemId = cartKey.toString().match(/^\d+$/); // Check if it's numeric ID
        
        let url = isCartItemId 
            ? `/cart/remove/${cartKey}` 
            : `/cart/remove/${cartKey}`; // For guests, cartKey is the productId_dealId
        
        fetch(url, {
            method: 'POST',
            body: formData
        })
        .then(response => {
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            return response.json();
        })
        .then(data => {
            if (data.success) {
                showNotification('Item removed from cart');
                // Remove the row from table
                const row = document.getElementById(`cart-row-${cartKey}`);
                if (row) {
                    row.remove();
                }
                
                // Update cart count in navbar
                updateCartCount();
                
                // Update cart totals
                updateCartTotals();
                
                // Reload if cart is now empty
                if (document.querySelectorAll('[id^="cart-row-"]').length === 0) {
                    setTimeout(() => {
                        window.location.reload();
                    }, 500);
                }
            } else {
                showNotification(data.message || 'Failed to remove item', 'error');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showNotification('An error occurred. Please try again.', 'error');
        });
    }
    
    // Clear entire cart
    function clearCart() {
        if (!confirm('Clear all items from cart?')) {
            return;
        }
        
        const csrfToken = document.querySelector('meta[name="csrf-token"]').getAttribute('content');
        
        // Use FormData
        const formData = new FormData();
        formData.append('_token', csrfToken);
        
        fetch('/cart/clear', {
            method: 'POST',
            body: formData
        })
        .then(response => {
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            return response.json();
        })
        .then(data => {
            if (data.success) {
                showNotification('Cart cleared');
                // Remove all rows
                document.querySelectorAll('[id^="cart-row-"]').forEach(row => row.remove());
                updateCartCount();
                // Reload to show empty cart message
                setTimeout(() => {
                    window.location.reload();
                }, 500);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showNotification('An error occurred. Please try again.', 'error');
        });
    }
    
    // Update cart count in navbar
    function updateCartCount() {
        fetch('/cart/count')
            .then(response => {
                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }
                return response.json();
            })
            .then(data => {
                const cartCountElement = document.getElementById('cart-count');
                if (cartCountElement) {
                    cartCountElement.textContent = data.cart_count || 0;
                }
            })
            .catch(error => {
                console.error('Error fetching cart count:', error);
            });
    }

    function redirectToLogin() {
        // Store current URL in session so we can redirect back after login
        const csrfToken = document.querySelector('meta[name="csrf-token"]').getAttribute('content');
        const formData = new FormData();
        formData.append('_token', csrfToken);
        formData.append('redirect_url', window.location.href);
        
        fetch('/store-redirect-url', {
            method: 'POST',
            body: formData
        })
        .then(() => {
            // Redirect to login page
            window.location.href = '{{ route("login") }}';
        })
        .catch(error => {
            console.error('Error:', error);
            // Fallback: redirect without storing URL
            window.location.href = '{{ route("login") }}';
        });
    }
    
    // FIXED: Add event listener for manual input changes
    document.addEventListener('DOMContentLoaded', function() {
        document.querySelectorAll('input[id^="quantity-input-"]').forEach(input => {
            input.addEventListener('change', function() {
                const cartKey = this.id.replace('quantity-input-', '');
                const quantity = parseInt(this.value) || 1;
                
                // Update this item's total
                updateSingleItemTotal(cartKey);
                
                // Update cart totals
                updateCartTotals();
                
                // Save to server
                saveQuantityToServer(cartKey, quantity);
            });
            
            // Also handle blur event
            input.addEventListener('blur', function() {
                const cartKey = this.id.replace('quantity-input-', '');
                const quantity = parseInt(this.value) || 1;
                
                // Make sure quantity is within bounds
                const min = parseInt(this.getAttribute('min')) || 1;
                const max = parseInt(this.getAttribute('max')) || 999;
                
                if (quantity < min) {
                    this.value = min;
                    updateSingleItemTotal(cartKey);
                    updateCartTotals();
                } else if (quantity > max) {
                    this.value = max;
                    showNotification(`Maximum ${max} items available`, 'error');
                    updateSingleItemTotal(cartKey);
                    updateCartTotals();
                }
            });
        });
        
        // Initialize all item totals
        document.querySelectorAll('[id^="cart-row-"]').forEach(row => {
            const cartKey = row.id.replace('cart-row-', '');
            updateSingleItemTotal(cartKey);
        });
        
        // Initialize cart totals
        updateCartTotals();
    });
</script>
@endpush