@if(isset($product))
    <form action="{{ route('admin.products.update', $product) }}" method="POST" enctype="multipart/form-data" class="space-y-6">
    @method('PUT')
@else
    <form action="{{ route('admin.products.store') }}" method="POST" enctype="multipart/form-data" class="space-y-6">
@endif
    @csrf

    <!-- Form Errors -->
    @if($errors->any())
        <div class="rounded-lg bg-red-50 p-4 border border-red-200">
            <div class="flex">
                <svg class="h-5 w-5 text-red-400 flex-shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"/>
                </svg>
                <div class="ml-3">
                    <h3 class="text-sm font-medium text-red-800">Please correct the following errors:</h3>
                    <div class="mt-2 text-sm text-red-700">
                        <ul class="list-disc pl-5 space-y-1">
                            @foreach($errors->all() as $error)
                                <li>{{ $error }}</li>
                            @endforeach
                        </ul>
                    </div>
                </div>
            </div>
        </div>
    @endif

    <!-- Two Column Grid Layout -->
    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
        <!-- Left Column -->
        <div class="space-y-5">
            <!-- Product Name -->
            <div>
                <label for="name" class="block text-sm font-medium text-gray-700 mb-1.5">
                    Product Name <span class="text-red-500">*</span>
                </label>
                <input type="text" 
                       name="name" 
                       id="name" 
                       value="{{ old('name', $product->name ?? '') }}"
                       required
                       class="mt-1 block w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0A9961] focus:border-[#0A9961] transition-colors duration-200 @error('name') border-red-500 @enderror"
                       placeholder="Enter product name">
                @error('name')
                    <p class="mt-1.5 text-sm text-red-600">{{ $message }}</p>
                @enderror
            </div>

            <!-- Slug -->
            <div>
                <label for="slug" class="block text-sm font-medium text-gray-700 mb-1.5">
                    URL Slug <span class="text-red-500">*</span>
                </label>
                <input type="text" 
                       name="slug" 
                       id="slug" 
                       value="{{ old('slug', $product->slug ?? '') }}"
                       required
                       class="mt-1 block w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0A9961] focus:border-[#0A9961] transition-colors duration-200 @error('slug') border-red-500 @enderror"
                       placeholder="product-url-slug">
                <p class="mt-1.5 text-xs text-gray-500">Used in product URLs. Auto-generates from name.</p>
                @error('slug')
                    <p class="mt-1.5 text-sm text-red-600">{{ $message }}</p>
                @enderror
            </div>

            <!-- Category -->
            <div>
                <label for="category_id" class="block text-sm font-medium text-gray-700 mb-1.5">
                    Category <span class="text-red-500">*</span>
                </label>
                <select name="category_id" 
                        id="category_id"
                        required
                        class="mt-1 block w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0A9961] focus:border-[#0A9961] transition-colors duration-200 @error('category_id') border-red-500 @enderror">
                    <option value="">Select category</option>
                    @foreach($categories as $category)
                        <option value="{{ $category->id }}" {{ old('category_id', $product->category_id ?? '') == $category->id ? 'selected' : '' }}>
                            {{ $category->name }}
                        </option>
                    @endforeach
                </select>
                @error('category_id')
                    <p class="mt-1.5 text-sm text-red-600">{{ $message }}</p>
                @enderror
            </div>

            <!-- Price -->
            <div>
                <label for="price" class="block text-sm font-medium text-gray-700 mb-1.5">
                    Price (Ksh) <span class="text-red-500">*</span>
                </label>
                <div class="mt-1 relative rounded-lg">
                    <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                        <span class="text-gray-500 sm:text-sm">Ksh</span>
                    </div>
                    <input type="number" 
                           name="price" 
                           id="price" 
                           step="0.01"
                           min="0"
                           value="{{ old('price', $product->price ?? '') }}"
                           required
                           class="block w-full pl-10 pr-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0A9961] focus:border-[#0A9961] transition-colors duration-200 @error('price') border-red-500 @enderror"
                           placeholder="0.00">
                </div>
                @error('price')
                    <p class="mt-1.5 text-sm text-red-600">{{ $message }}</p>
                @enderror
            </div>
        </div>

        <!-- Right Column -->
        <div class="space-y-5">
            <!-- Brand -->
            <div>
                <label for="brand" class="block text-sm font-medium text-gray-700 mb-1.5">
                    Brand
                </label>
                <input type="text" 
                       name="brand" 
                       id="brand" 
                       value="{{ old('brand', $product->brand ?? '') }}"
                       class="mt-1 block w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0A9961] focus:border-[#0A9961] transition-colors duration-200 @error('brand') border-red-500 @enderror"
                       placeholder="Enter brand name">
                @error('brand')
                    <p class="mt-1.5 text-sm text-red-600">{{ $message }}</p>
                @enderror
            </div>

            <!-- Stock -->
            <div>
                <label for="stock" class="block text-sm font-medium text-gray-700 mb-1.5">
                    Stock Quantity <span class="text-red-500">*</span>
                </label>
                <input type="number" 
                       name="stock" 
                       id="stock" 
                       min="0"
                       value="{{ old('stock', $product->stock ?? '') }}"
                       required
                       class="mt-1 block w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0A9961] focus:border-[#0A9961] transition-colors duration-200 @error('stock') border-red-500 @enderror"
                       placeholder="Enter quantity">
                @error('stock')
                    <p class="mt-1.5 text-sm text-red-600">{{ $message }}</p>
                @enderror
            </div>

            <!-- SKU -->
            <div>
                <label for="sku" class="block text-sm font-medium text-gray-700 mb-1.5">
                    SKU (Stock Keeping Unit)
                </label>
                <input type="text" 
                       name="sku" 
                       id="sku" 
                       value="{{ old('sku', $product->sku ?? '') }}"
                       class="mt-1 block w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0A9961] focus:border-[#0A9961] transition-colors duration-200 @error('sku') border-red-500 @enderror"
                       placeholder="e.g., PROD-001">
                @error('sku')
                    <p class="mt-1.5 text-sm text-red-600">{{ $message }}</p>
                @enderror
            </div>

            <!-- Status -->
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1.5">
                    Status
                </label>
                <div class="mt-1 flex items-center space-x-4">
                    <label class="inline-flex items-center">
                        <input type="radio" name="is_active" value="1" {{ old('is_active', $product->is_active ?? 1) ? 'checked' : '' }}
                               class="h-4 w-4 text-[#0A9961] focus:ring-[#0A9961] border-gray-300">
                        <span class="ml-2 text-sm text-gray-700">Active</span>
                    </label>
                    <label class="inline-flex items-center">
                        <input type="radio" name="is_active" value="0" {{ !old('is_active', $product->is_active ?? 1) ? 'checked' : '' }}
                               class="h-4 w-4 text-gray-600 focus:ring-gray-500 border-gray-300">
                        <span class="ml-2 text-sm text-gray-700">Inactive</span>
                    </label>
                </div>
                @error('is_active')
                    <p class="mt-1.5 text-sm text-red-600">{{ $message }}</p>
                @enderror
            </div>
        </div>
    </div>

    <!-- Description -->
    <div>
        <label for="description" class="block text-sm font-medium text-gray-700 mb-1.5">
            Description
        </label>
        <textarea name="description" 
                  id="description" 
                  rows="4"
                  class="mt-1 block w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0A9961] focus:border-[#0A9961] transition-colors duration-200 @error('description') border-red-500 @enderror"
                  placeholder="Describe your product in detail...">{{ old('description', $product->description ?? '') }}</textarea>
        <div class="mt-1.5 flex justify-between items-center">
            <p class="text-xs text-gray-500">Maximum 1000 characters</p>
            <span id="charCount" class="text-xs text-gray-500">0 characters</span>
        </div>
        @error('description')
            <p class="mt-1.5 text-sm text-red-600">{{ $message }}</p>
        @enderror
    </div>

    <!-- Image Upload -->
    <div>
        <label class="block text-sm font-medium text-gray-700 mb-1.5">
            Product Image
        </label>
        
        <!-- Current Image Preview -->
        @if(isset($product) && $product->image)
            <div class="mb-4 p-4 border border-gray-200 rounded-lg bg-gray-50">
                <p class="text-sm font-medium text-gray-700 mb-2">Current Image:</p>
                <div class="flex items-center space-x-4">
                    <img src="{{ asset($product->image) }}" 
                         alt="{{ $product->name }}" 
                         class="h-24 w-24 object-cover rounded-lg border border-gray-300">
                    <div class="text-sm text-gray-600">
                        <p>Upload a new image to replace this one</p>
                        <label class="inline-flex items-center mt-2">
                            <input type="checkbox" name="remove_image" value="1" 
                                   class="h-4 w-4 text-red-600 focus:ring-red-500 border-gray-300 rounded">
                            <span class="ml-2 text-sm text-red-600">Remove current image</span>
                        </label>
                    </div>
                </div>
            </div>
        @endif

        <!-- File Upload -->
        <div class="mt-2">
            <div id="uploadContainer" class="flex justify-center px-6 pt-5 pb-6 border-2 border-gray-300 border-dashed rounded-lg hover:border-gray-400 transition-colors duration-200">
                <div class="space-y-1 text-center">
                    <svg class="mx-auto h-12 w-12 text-gray-400" stroke="currentColor" fill="none" viewBox="0 0 48 48">
                        <path d="M28 8H12a4 4 0 00-4 4v20m32-12v8m0 0v8a4 4 0 01-4 4H12a4 4 0 01-4-4v-4m32-4l-3.172-3.172a4 4 0 00-5.656 0L28 28M8 32l9.172-9.172a4 4 0 015.656 0L28 28m0 0l4 4m4-24h8m-4-4v8m-12 4h.02" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                    </svg>
                    <div class="flex text-sm text-gray-600 justify-center">
                        <label for="image" class="relative cursor-pointer rounded-md font-medium text-[#0A9961] hover:text-[#088552] focus-within:outline-none transition-colors duration-200">
                            <span>Upload a file</span>
                            <input id="image" name="image" type="file" class="sr-only" accept="image/*">
                        </label>
                        <p class="pl-1">or drag and drop</p>
                    </div>
                    <p class="text-xs text-gray-500">
                        PNG, JPG, GIF up to 5MB
                    </p>
                </div>
            </div>
            
            <!-- Image Preview Container -->
            <div id="imagePreviewContainer"></div>
        </div>
        @error('image')
            <p class="mt-1.5 text-sm text-red-600">{{ $message }}</p>
        @enderror
    </div>

    <!-- SEO Settings -->
    <div class="p-4 bg-gray-50 rounded-lg border border-gray-200">
        <h3 class="text-sm font-medium text-gray-900 mb-3">SEO Settings (Optional)</h3>
        
        <div class="space-y-4">
            <div>
                <label for="meta_title" class="block text-xs font-medium text-gray-700 mb-1">
                    Meta Title
                </label>
                <input type="text" id="meta_title" name="meta_title" value="{{ old('meta_title', $product->meta_title ?? '') }}"
                       class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:outline-none focus:ring-1 focus:ring-[#0A9961] focus:border-transparent"
                       placeholder="Meta title for search engines">
            </div>
            
            <div>
                <label for="meta_description" class="block text-xs font-medium text-gray-700 mb-1">
                    Meta Description
                </label>
                <textarea id="meta_description" name="meta_description" rows="2"
                          class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:outline-none focus:ring-1 focus:ring-[#0A9961] focus:border-transparent"
                          placeholder="Meta description for search engines">{{ old('meta_description', $product->meta_description ?? '') }}</textarea>
            </div>
            
            <div>
                <label for="meta_keywords" class="block text-xs font-medium text-gray-700 mb-1">
                    Meta Keywords
                </label>
                <input type="text" id="meta_keywords" name="meta_keywords" value="{{ old('meta_keywords', $product->meta_keywords ?? '') }}"
                       class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:outline-none focus:ring-1 focus:ring-[#0A9961] focus:border-transparent"
                       placeholder="keyword1, keyword2, keyword3">
            </div>
        </div>
    </div>

    <!-- Form Actions -->
    <div class="pt-6 border-t border-gray-200">
        <div class="flex justify-end space-x-3">
            <a href="{{ route('admin.products.index') }}" 
               class="px-5 py-2.5 border border-gray-300 rounded-lg text-gray-700 font-medium hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-[#0A9961] transition-colors duration-200">
                Cancel
            </a>
            <button type="submit" 
                    class="px-5 py-2.5 bg-[#0A9961] hover:bg-[#088552] text-white font-medium rounded-lg focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-[#0A9961] transition-colors duration-200">
                {{ isset($product) ? 'Update Product' : 'Create Product' }}
            </button>
        </div>
    </div>
</form>

<script>
    document.addEventListener('DOMContentLoaded', function() {
        // Character counter for description
        const descriptionField = document.getElementById('description');
        const charCount = document.getElementById('charCount');
        
        if (descriptionField && charCount) {
            charCount.textContent = `${descriptionField.value.length} characters`;
            
            descriptionField.addEventListener('input', function() {
                const length = this.value.length;
                charCount.textContent = `${length} characters`;
                
                if (length > 1000) {
                    charCount.classList.remove('text-gray-500');
                    charCount.classList.add('text-red-500');
                } else {
                    charCount.classList.remove('text-red-500');
                    charCount.classList.add('text-gray-500');
                }
            });
        }
        
        // Auto-generate slug from product name
        document.getElementById('name').addEventListener('input', function(e) {
            const slugInput = document.getElementById('slug');
            if (!slugInput.value || slugInput.value === '{{ $product->slug ?? "" }}') {
                const slug = e.target.value
                    .toLowerCase()
                    .replace(/[^\w\s-]/g, '')  // Remove special characters
                    .replace(/\s+/g, '-')      // Replace spaces with hyphens
                    .replace(/--+/g, '-')      // Remove double hyphens
                    .replace(/^-+|-+$/g, '');  // Remove leading/trailing hyphens
                slugInput.value = slug;
            }
        });
        
        // Image preview functionality
        const imageInput = document.getElementById('image');
        const previewContainer = document.getElementById('imagePreviewContainer');
        const uploadContainer = document.getElementById('uploadContainer');
        
        if (imageInput) {
            // Handle file selection
            imageInput.addEventListener('change', function(e) {
                const file = e.target.files[0];
                if (file) {
                    if (file.size > 5 * 1024 * 1024) { // 5MB limit
                        alert('File size must be less than 5MB');
                        this.value = '';
                        return;
                    }
                    
                    const reader = new FileReader();
                    reader.onload = function(e) {
                        // Create preview
                        previewContainer.innerHTML = `
                            <div class="mt-4 p-4 border border-green-200 rounded-lg bg-green-50">
                                <p class="text-sm font-medium text-green-800 mb-2">Image Preview:</p>
                                <div class="flex items-center space-x-4">
                                    <img src="${e.target.result}" class="h-24 w-24 object-cover rounded-lg border border-green-300">
                                    <div>
                                        <p class="text-sm text-green-700 font-medium">${file.name}</p>
                                        <p class="text-xs text-green-600">${(file.size / 1024 / 1024).toFixed(2)} MB</p>
                                        <button type="button" onclick="removeImagePreview()" class="mt-2 text-xs text-red-600 hover:text-red-800">
                                            Remove
                                        </button>
                                    </div>
                                </div>
                            </div>
                        `;
                        
                        // Hide upload container
                        uploadContainer.classList.add('hidden');
                    };
                    reader.readAsDataURL(file);
                }
            });
            
            // Drag and drop functionality
            ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
                uploadContainer.addEventListener(eventName, preventDefaults, false);
            });

            function preventDefaults(e) {
                e.preventDefault();
                e.stopPropagation();
            }

            ['dragenter', 'dragover'].forEach(eventName => {
                uploadContainer.addEventListener(eventName, highlight, false);
            });

            ['dragleave', 'drop'].forEach(eventName => {
                uploadContainer.addEventListener(eventName, unhighlight, false);
            });

            function highlight() {
                uploadContainer.classList.add('border-[#0A9961]', 'bg-green-50');
            }

            function unhighlight() {
                uploadContainer.classList.remove('border-[#0A9961]', 'bg-green-50');
            }

            uploadContainer.addEventListener('drop', handleDrop, false);

            function handleDrop(e) {
                const dt = e.dataTransfer;
                const files = dt.files;
                
                if (files.length > 0) {
                    imageInput.files = files;
                    imageInput.dispatchEvent(new Event('change'));
                }
            }
        }
        
        // Price formatting
        const priceField = document.getElementById('price');
        if (priceField) {
            priceField.addEventListener('blur', function() {
                if (this.value && !isNaN(this.value)) {
                    this.value = parseFloat(this.value).toFixed(2);
                }
            });
        }

        // Auto-fill SEO fields
        document.getElementById('name').addEventListener('blur', function() {
            const name = this.value;
            const metaTitle = document.getElementById('meta_title');
            const metaDescription = document.getElementById('meta_description');
            
            if (name && !metaTitle.value) {
                metaTitle.value = name + ' | ' + '{{ config("app.name", "Our Store") }}';
            }
            
            if (name && !metaDescription.value) {
                const desc = 'Buy ' + name + ' at the best price. Free shipping available.';
                metaDescription.value = desc;
            }
        });
    });
    
    // Function to remove image preview
    function removeImagePreview() {
        const previewContainer = document.getElementById('imagePreviewContainer');
        const uploadContainer = document.getElementById('uploadContainer');
        const imageInput = document.getElementById('image');
        
        previewContainer.innerHTML = '';
        uploadContainer.classList.remove('hidden');
        imageInput.value = '';
    }
</script>
