<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class SystemSetting extends Model
{
    use HasFactory;

    protected $fillable = ['key', 'value', 'type', 'description'];

    /**
     * Get a setting value
     */
    public static function getValue($key, $default = null)
    {
        $setting = self::where('key', $key)->first();
        
        if (!$setting) {
            return $default;
        }

        // Convert value based on type
        switch ($setting->type) {
            case 'boolean':
                return (bool) $setting->value;
            case 'integer':
                return (int) $setting->value;
            case 'json':
                return json_decode($setting->value, true);
            default:
                return $setting->value;
        }
    }

    /**
     * Set a setting value
     */
    public static function setValue($key, $value, $type = 'string', $description = null)
    {
        $setting = self::updateOrCreate(
            ['key' => $key],
            [
                'value' => $value,
                'type' => $type,
                'description' => $description
            ]
        );

        return $setting;
    }

    /**
     * Get all system settings as an array
     */
    public static function getAllSettings()
    {
        $settings = self::all();
        $result = [];
        
        foreach ($settings as $setting) {
            // Convert value based on type
            switch ($setting->type) {
                case 'boolean':
                    $result[$setting->key] = (bool) $setting->value;
                    break;
                case 'integer':
                    $result[$setting->key] = (int) $setting->value;
                    break;
                case 'json':
                    $result[$setting->key] = json_decode($setting->value, true);
                    break;
                default:
                    $result[$setting->key] = $setting->value;
            }
        }
        
        return $result;
    }

    /**
     * Save multiple system settings
     */
    public static function saveSettings(array $settings)
    {
        // Define the type for each setting
        $settingTypes = [
            'site_name' => 'string',
            'site_email' => 'string',
            'site_phone' => 'string',
            'site_address' => 'string',
            'currency' => 'string',
            'timezone' => 'string',
            'mpesa_enabled' => 'boolean',
            'cash_on_delivery_enabled' => 'boolean',
            'direct_paybill_enabled' => 'boolean',
            'paybill_number' => 'string',
            'paybill_account_number' => 'string',
            'paybill_instructions' => 'text',
        ];
        
        foreach ($settings as $key => $value) {
            $type = $settingTypes[$key] ?? 'string';
            $description = self::getSettingDescription($key);
            
            // For boolean checkboxes, ensure proper value
            if ($type === 'boolean') {
                $value = isset($settings[$key]) && $settings[$key] == '1' ? '1' : '0';
            }
            
            self::setValue($key, $value, $type, $description);
        }
        
        return true;
    }

    /**
     * Get description for a setting
     */
    private static function getSettingDescription($key)
    {
        $descriptions = [
            'site_name' => 'Name of the website/store',
            'site_email' => 'Default email for the site',
            'site_phone' => 'Contact phone number',
            'site_address' => 'Business physical address',
            'currency' => 'Default currency for the store',
            'timezone' => 'Default timezone for the store',
            'mpesa_enabled' => 'Enable/disable M-Pesa STK Push payments',
            'cash_on_delivery_enabled' => 'Enable/disable Cash on Delivery payments',
            'direct_paybill_enabled' => 'Enable/disable Direct Paybill payments',
            'paybill_number' => 'Bank Paybill number for manual payments',
            'paybill_account_number' => 'Account number for paybill payments',
            'paybill_instructions' => 'Instructions shown to customers for paybill payments',
        ];
        
        return $descriptions[$key] ?? 'System setting';
    }

    /**
     * Check if Direct Paybill is enabled
     */
    public static function isDirectPaybillEnabled()
    {
        return self::getValue('direct_paybill_enabled', false);
    }

    /**
     * Get Paybill details
     */
    public static function getPaybillDetails()
    {
        return [
            'enabled' => self::getValue('direct_paybill_enabled', false),
            'paybill_number' => self::getValue('paybill_number', ''),
            'account_number' => self::getValue('paybill_account_number', ''),
            'instructions' => self::getValue('paybill_instructions', ''),
        ];
    }

    /**
     * Get payment methods availability
     */
    public static function getAvailablePaymentMethods()
    {
        return [
            'mpesa' => self::getValue('mpesa_enabled', false),
            'cod' => self::getValue('cash_on_delivery_enabled', false),
            'paybill' => self::getValue('direct_paybill_enabled', false),
        ];
    }
}