<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use Illuminate\Http\Request;

class OrderController extends Controller
{
    public function index()
    {
        $orders = Order::with(['user', 'items'])
            ->latest()
            ->paginate(15);
        
        // Get counts for statistics
        $pendingCount = Order::where('status', 'pending')->count();
        $completedCount = Order::where('status', 'delivered')->count();
        $processingCount = Order::where('status', 'processing')->orWhere('status', 'shipped')->count();
        $cancelledCount = Order::where('status', 'cancelled')->orWhere('status', 'refunded')->count();
        
        return view('admin.orders.index', compact('orders', 'pendingCount', 'completedCount', 'processingCount', 'cancelledCount'));
    }

    public function show(Order $order)
    {
        // Load Mpesa transactions with the order
        $order->load(['user', 'items.product', 'mpesaTransactions']);
        
        return view('admin.orders.show', compact('order'));
    }

    public function edit(Order $order)
    {
        return view('admin.orders.edit', compact('order'));
    }

    public function update(Request $request, Order $order)
    {
        $validated = $request->validate([
            'status' => 'required|in:pending,processing,shipped,delivered,cancelled,refunded',
            'payment_status' => 'required|in:pending,paid,failed',
            'notes' => 'nullable|string',
        ]);

        $order->update($validated);

        return redirect()->route('admin.orders.show', $order)
            ->with('success', 'Order updated successfully!');
    }

    public function destroy(Order $order)
    {
        $order->delete();
        
        return redirect()->route('admin.orders.index')
            ->with('success', 'Order deleted successfully!');
    }
}