@extends('layouts.app')

@section('title', 'Finepoint Electronics - Premium Electronics Store')

@push('styles')
<style>
    .product-card { transition: all 0.2s ease-in-out; }
    .product-card:hover { transform: translateY(-2px); box-shadow: 0 10px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04); }
    .line-clamp-2 { display: -webkit-box; -webkit-line-clamp: 2; -webkit-box-orient: vertical; overflow: hidden; }
    
    /* Hero gradient */
    .hero-gradient {
        background: linear-gradient(135deg, #0A9961 0%, #10B981 100%);
    }
</style>
@endpush

@section('content')
<!-- Hero Section -->
<div class="hero-gradient text-white">
    <div class="max-w-7xl mx-auto px-2 py-4 sm:py-2">
        <div class="text-center">
            <h1 class="text-3xl sm:text-4xl font-bold mb-3">Welcome to Finepoint Electronics</h1>
            <p class="text-gray-100 mb-6 text-sm sm:text-base">Premium electronics at affordable prices</p>
            <a href="{{ route('products.public') }}" 
               class="inline-block bg-white text-[#0A9961] hover:bg-gray-100 font-medium py-2 px-6 rounded transition duration-200 no-underline">
                Shop Now
            </a>
        </div>
    </div>
</div>

<!-- Categories -->
<div class="max-w-7xl mx-auto px-4 py-8">
    <h2 class="text-2xl font-bold text-gray-900 mb-6 text-center">Shop Categories</h2>
    @if($categories->count() > 0)
    <div class="grid grid-cols-3 sm:grid-cols-3 md:grid-cols-4 lg:grid-cols-5 xl:grid-cols-6 gap-4">
        @foreach($categories as $category)
        <a href="{{ route('category.products', $category) }}" 
           class="group relative aspect-square rounded-xl overflow-hidden shadow-lg hover:shadow-xl transition-all duration-300 no-underline block">
            <!-- Category Image Background -->
            @if($category->image_url)
                <img src="{{ $category->image_url }}" 
                     alt="{{ $category->name }}"
                     class="w-full h-full object-cover group-hover:scale-110 transition-transform duration-500">
            @else
                <div class="w-full h-full bg-gradient-to-br from-[#0A9961] to-[#10B981] flex items-center justify-center">
                    <i class="fas fa-folder text-white text-3xl"></i>
                </div>
            @endif
            
            <!-- Gradient Overlay -->
            <div class="absolute inset-0 bg-gradient-to-t from-black/30 via-gray/40 to-transparent"></div>
            
            <!-- Category Content -->
            <div class="absolute bottom-0 left-0 right-0 p-3">
                <h3 class="text-sm font-bold flex items-center justify-center bg-gradient-to-r from-gray-200 to-gray-200 text-[#0A9961] mb-1 truncate">{{ $category->name }}</h3>
                @if($category->products_count > 0)
                    <div class="flex items-center justify-between">
                        <span class="text-white/90 text-xs">
                            {{ $category->products_count }}
                        </span>
                        <span class="text-white text-xs font-medium group-hover:translate-x-1 transition-transform duration-200">
                            <i class="fas fa-arrow-right"></i>
                        </span>
                    </div>
                @else
                    <span class="text-white/70 text-xs">Coming Soon</span>
                @endif
            </div>
        </a>
        @endforeach
    </div>
    @else
    <div class="text-center py-12 bg-gray-50 rounded-xl">
        <i class="fas fa-folder-open text-4xl text-gray-300 mb-4"></i>
        <p class="text-gray-500 text-lg">No categories available yet.</p>
    </div>
    @endif
</div>

<!-- Deals Section -->
@if(isset($featuredDeals) && $featuredDeals->count() > 0)
<div class="bg-gradient-to-r from-[#0A9961] to-[#10B981] text-white py-8">
    <div class="max-w-7xl mx-auto px-4">
        <div class="flex justify-between items-center mb-6">
            <h2 class="text-2xl font-bold">🔥 Hot Deals</h2>
            <a href="{{ route('deals.index') }}" class="text-white hover:text-gray-200 text-sm no-underline flex items-center">
                View All Deals <i class="fas fa-arrow-right ml-2"></i>
            </a>
        </div>
        
        <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
            @foreach($featuredDeals as $deal)
            <a href="{{ route('deals.show', $deal->slug) }}" class="block no-underline">
                <div class="bg-white/10 backdrop-blur-sm rounded-xl p-5 hover:bg-white/20 transition-all duration-300">
                    <div class="flex items-center justify-between mb-3">
                        <span class="text-xs font-bold bg-red-500 px-3 py-1 rounded-full">
                            <i class="fas fa-clock mr-1"></i> {{ $deal->end_date->diffForHumans(['parts' => 1]) }}
                        </span>
                        @if($deal->type == 'percentage')
                        <span class="text-lg font-bold">{{ $deal->discount_value }}% OFF</span>
                        @elseif($deal->type == 'fixed')
                        <span class="text-lg font-bold">Ksh {{ number_format($deal->discount_value) }} OFF</span>
                        @endif
                    </div>
                    
                    <h3 class="text-xl font-bold mb-2">{{ $deal->title }}</h3>
                    <p class="text-gray-200 text-sm mb-4 line-clamp-2">{{ $deal->description }}</p>
                    
                    <div class="flex items-center">
                        <div class="flex -space-x-2">
                            @foreach($deal->products->take(3) as $product)
                            @if($product->image)
                            <img src="{{ $product->image_url }}" 
                                 alt="{{ $product->name }}"
                                 class="w-8 h-8 rounded-full border-2 border-white object-cover">
                            @else
                            <div class="w-8 h-8 rounded-full bg-gray-300 border-2 border-white flex items-center justify-center">
                                <i class="fas fa-box text-gray-600 text-xs"></i>
                            </div>
                            @endif
                            @endforeach
                        </div>
                        <span class="ml-3 text-sm">
                            {{ $deal->products->count() }} products
                        </span>
                        <span class="ml-auto text-sm font-medium">
                            Shop Now <i class="fas fa-arrow-right ml-1"></i>
                        </span>
                    </div>
                </div>
            </a>
            @endforeach
        </div>
    </div>
</div>
@endif

<!-- Featured Products - COMPACT VERSION -->
<div class="bg-white py-6">
    <div class="max-w-7xl mx-auto px-4">
        <div class="flex justify-between items-center mb-4">
            <h2 class="text-lg font-bold text-gray-900">Featured Products</h2>
            <a href="{{ route('products.public') }}" class="text-[#0A9961] hover:text-[#088552] text-sm no-underline">
                View All →
            </a>
        </div>
        
        @if($featuredProducts->count() > 0)
        <div class="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 lg:grid-cols-5 xl:grid-cols-6 gap-4">
            @foreach($featuredProducts as $product)
            <div class="bg-white border border-gray-200 rounded-lg p-3 product-card">
                <!-- Product Image -->
                <a href="{{ route('products.public.show', $product) }}" class="block mb-2 no-underline">
                    @if($product->image)
                        <div class="relative w-full h-32 bg-gray-50 rounded overflow-hidden">
                            <img src="{{ asset($product->image) }}" 
                                 alt="{{ $product->name }}"
                                 class="w-full h-full object-contain p-1">
                        </div>
                    @else
                        <div class="w-full h-32 bg-gray-100 rounded flex items-center justify-center">
                            <i class="fas fa-box text-gray-300 text-xl"></i>
                        </div>
                    @endif
                </a>
                
                <!-- Product Info -->
                <h3 class="font-medium text-gray-900 text-sm mb-1 line-clamp-2 h-10">
                    <a href="{{ route('products.public.show', $product) }}" class="hover:text-[#0A9961] no-underline">
                        {{ $product->name }}
                    </a>
                </h3>
                
                <!-- Brand -->
                @if($product->brand)
                <div class="mb-2">
                    <span class="text-xs text-gray-500 truncate block">{{ $product->brand }}</span>
                </div>
                @endif
                
                <!-- Price -->
                <div class="mb-2">
                    <span class="text-base font-bold text-[#0A9961]">
                        Ksh {{ number_format($product->price) }}
                    </span>
                </div>
                
                <!-- Stock Status Badge -->
                <div class="flex items-center justify-between">
                    @if($product->stock > 0)
                        @if($product->stock <= 10)
                            <span class="text-xs text-yellow-600 bg-yellow-50 px-2 py-1 rounded">Low Stock</span>
                        @else
                            <span class="text-xs text-green-600 bg-green-50 px-2 py-1 rounded">In Stock</span>
                        @endif
                    @else
                        <span class="text-xs text-red-600 bg-red-50 px-2 py-1 rounded">Out of Stock</span>
                    @endif
                    
                    <button onclick="addToCart(event, {{ $product->id }}, '{{ addslashes($product->name) }}', {{ $product->price }}, {{ $product->stock }})" 
                            class="text-[#0A9961] hover:text-[#088552] p-1 {{ $product->stock <= 0 ? 'opacity-50 cursor-not-allowed' : '' }} bg-transparent border-none cursor-pointer"
                            {{ $product->stock <= 0 ? 'disabled' : '' }}>
                        <i class="fas fa-cart-plus"></i>
                    </button>
                </div>
            </div>
            @endforeach
        </div>
        @else
        <div class="text-center py-8">
            <i class="fas fa-box-open text-3xl text-gray-300 mb-3"></i>
            <p class="text-gray-500">No products available yet.</p>
            @auth
                @if(auth()->user()->isAdmin())
                    <a href="{{ route('admin.products.create') }}" class="mt-3 inline-block text-[#0A9961] hover:text-[#088552] text-sm no-underline">
                        Add your first product →
                    </a>
                @endif
            @endauth
        </div>
        @endif
    </div>
</div>

<!-- Features -->
<div class="bg-gray-50 py-6">
    <div class="max-w-7xl mx-auto px-4">
        <div class="grid grid-cols-2 md:grid-cols-4 gap-4">
            <div class="text-center">
                <div class="w-8 h-8 rounded-full bg-[#0A9961] flex items-center justify-center mx-auto mb-2">
                    <i class="fas fa-shipping-fast text-white text-sm"></i>
                </div>
                <h3 class="font-medium text-gray-900 text-sm">Free Shipping</h3>
                <p class="text-xs text-gray-600">Over Ksh 10,000</p>
            </div>
            <div class="text-center">
                <div class="w-8 h-8 rounded-full bg-[#0A9961] flex items-center justify-center mx-auto mb-2">
                    <i class="fas fa-undo text-white text-sm"></i>
                </div>
                <h3 class="font-medium text-gray-900 text-sm">30-Day Returns</h3>
                <p class="text-xs text-gray-600">Easy returns</p>
            </div>
            <div class="text-center">
                <div class="w-8 h-8 rounded-full bg-[#0A9961] flex items-center justify-center mx-auto mb-2">
                    <i class="fas fa-shield-alt text-white text-sm"></i>
                </div>
                <h3 class="font-medium text-gray-900 text-sm">1 Year Warranty</h3>
                <p class="text-xs text-gray-600">On all products</p>
            </div>
            <div class="text-center">
                <div class="w-8 h-8 rounded-full bg-[#0A9961] flex items-center justify-center mx-auto mb-2">
                    <i class="fas fa-headset text-white text-sm"></i>
                </div>
                <h3 class="font-medium text-gray-900 text-sm">24/7 Support</h3>
                <p class="text-xs text-gray-600">Dedicated help</p>
            </div>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
    // Function to show notifications
    function showNotification(message, type = 'success') {
        // Create notification element
        const notification = document.createElement('div');
        const bgColor = type === 'success' ? 'bg-green-500' : 'bg-red-500';
        const icon = type === 'success' ? 'fa-check-circle' : 'fa-exclamation-circle';
        
        notification.className = `${bgColor} text-white px-4 py-3 rounded-lg shadow-lg z-50 fixed top-4 right-4 transition-all duration-300 transform translate-x-full`;
        notification.innerHTML = `
            <div class="flex items-center">
                <i class="fas ${icon} mr-3"></i>
                <span>${message}</span>
                <button onclick="this.parentElement.parentElement.remove()" class="ml-4 text-white hover:text-gray-200">
                    <i class="fas fa-times"></i>
                </button>
            </div>
        `;
        
        // Add to document
        document.body.appendChild(notification);
        
        // Animate in
        setTimeout(() => {
            notification.classList.remove('translate-x-full');
            notification.classList.add('translate-x-0');
        }, 10);
        
        // Remove after 4 seconds
        setTimeout(() => {
            notification.classList.remove('translate-x-0');
            notification.classList.add('translate-x-full');
            setTimeout(() => {
                if (notification.parentNode) {
                    notification.parentNode.removeChild(notification);
                }
            }, 300);
        }, 4000);
    }
    
    // Main function to add product to cart
    function addToCart(event, productId, productName, price, stock) {
        event.preventDefault(); // Prevent any default behavior
        event.stopPropagation(); // Stop event bubbling
        
        // Check if product is in stock
        if (stock <= 0) {
            showNotification('This product is out of stock', 'error');
            return;
        }
        
        // Show loading state on the button
        const button = event.target.closest('button');
        const originalHTML = button.innerHTML;
        button.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
        button.disabled = true;
        
        // Get CSRF token from meta tag
        const csrfToken = document.querySelector('meta[name="csrf-token"]').getAttribute('content');
        
        // Prepare data for the request
        const data = {
            product_id: productId,
            quantity: 1,
            _token: csrfToken
        };
        
        // Make actual API call
        fetch('{{ route("cart.add") }}', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json',
                'X-CSRF-TOKEN': csrfToken
            },
            body: JSON.stringify(data)
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showNotification('✓ Added to cart: ' + productName);
                updateCartCount(data.cart_count);
            } else {
                showNotification(data.message || 'Failed to add to cart', 'error');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showNotification('An error occurred. Please try again.', 'error');
        })
        .finally(() => {
            // Restore button state
            button.innerHTML = originalHTML;
            button.disabled = false;
        });
    }
    
    // Function to update cart count
    function updateCartCount(count = null) {
        if (count !== null) {
            // If count is provided, update directly
            setCartCount(count);
        } else {
            // Fetch current cart count from server
            fetch('{{ route("cart.count") }}')
                .then(response => response.json())
                .then(data => {
                    setCartCount(data.cart_count);
                })
                .catch(error => {
                    console.error('Error fetching cart count:', error);
                });
        }
    }
    
    // Helper function to set cart count in the UI
    function setCartCount(count) {
        const cartCountElement = document.getElementById('cart-count');
        if (cartCountElement) {
            cartCountElement.textContent = count;
        }
    }
    
    // Initialize cart count on page load
    document.addEventListener('DOMContentLoaded', function() {
        updateCartCount();
        
        // Add event listeners to all product links to prevent cart button interference
        document.querySelectorAll('.product-card a').forEach(link => {
            link.addEventListener('click', function(e) {
                // Allow normal navigation for product links
                return true;
            });
        });
    });
</script>
@endpush