@extends('layouts.admin-layout')

@section('content')
<div class="container-fluid px-4">
    <div class="flex justify-between items-center mb-6">
        <div>
            <h1 class="text-2xl font-bold text-gray-900">Edit Deal</h1>
            <p class="text-gray-600 text-sm">Update deal information and settings</p>
        </div>
        <a href="{{ route('admin.deals.index') }}" 
           class="text-gray-600 hover:text-gray-900 no-underline flex items-center">
            <i class="fas fa-arrow-left mr-2"></i> Back to Deals
        </a>
    </div>

    <div class="bg-white rounded-lg shadow-sm overflow-hidden">
        <form action="{{ route('admin.deals.update', $deal) }}" method="POST" enctype="multipart/form-data">
            @csrf
            @method('PUT')
            
            <div class="p-6 space-y-6">
                <!-- Basic Information -->
                <div>
                    <h3 class="text-lg font-medium text-gray-900 mb-4">Basic Information</h3>
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <!-- Title -->
                        <div>
                            <label for="title" class="block text-sm font-medium text-gray-700 mb-2">Deal Title *</label>
                            <input type="text" 
                                   name="title" 
                                   id="title" 
                                   value="{{ old('title', $deal->title) }}"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0A9961] focus:border-[#0A9961]"
                                   required>
                            @error('title')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>
                        
                        <!-- Slug -->
                        <div>
                            <label for="slug" class="block text-sm font-medium text-gray-700 mb-2">Slug</label>
                            <input type="text" 
                                   name="slug" 
                                   id="slug" 
                                   value="{{ old('slug', $deal->slug) }}"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0A9961] focus:border-[#0A9961]"
                                   placeholder="auto-generates-if-empty">
                            <p class="mt-1 text-xs text-gray-500">URL-friendly version of the title</p>
                            @error('slug')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>
                    </div>
                    
                    <!-- Description -->
                    <div class="mt-6">
                        <label for="description" class="block text-sm font-medium text-gray-700 mb-2">Description</label>
                        <textarea name="description" 
                                  id="description" 
                                  rows="3"
                                  class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0A9961] focus:border-[#0A9961]">{{ old('description', $deal->description) }}</textarea>
                        @error('description')
                            <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                        @enderror
                    </div>
                </div>
                
                <!-- Deal Configuration -->
                <div>
                    <h3 class="text-lg font-medium text-gray-900 mb-4">Deal Configuration</h3>
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <!-- Type -->
                        <div>
                            <label for="type" class="block text-sm font-medium text-gray-700 mb-2">Deal Type *</label>
                            <select name="type" 
                                    id="type" 
                                    class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0A9961] focus:border-[#0A9961]"
                                    required>
                                <option value="">Select Type</option>
                                <option value="percentage" {{ old('type', $deal->type) == 'percentage' ? 'selected' : '' }}>Percentage Discount</option>
                                <option value="fixed" {{ old('type', $deal->type) == 'fixed' ? 'selected' : '' }}>Fixed Amount Discount</option>
                                <option value="bundle" {{ old('type', $deal->type) == 'bundle' ? 'selected' : '' }}>Bundle Deal</option>
                                <option value="special" {{ old('type', $deal->type) == 'special' ? 'selected' : '' }}>Special Offer</option>
                            </select>
                            @error('type')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>
                        
                        <!-- Discount Value -->
                        <div>
                            <label for="discount_value" class="block text-sm font-medium text-gray-700 mb-2">
                                <span id="discount_label">
                                    @if($deal->type == 'percentage') Discount Percentage *
                                    @elseif($deal->type == 'fixed') Discount Amount *
                                    @else Special Value
                                    @endif
                                </span>
                            </label>
                            <div class="relative">
                                <input type="number" 
                                       name="discount_value" 
                                       id="discount_value" 
                                       value="{{ old('discount_value', $deal->discount_value) }}"
                                       step="0.01"
                                       min="0"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0A9961] focus:border-[#0A9961] pr-10">
                                <div class="absolute inset-y-0 right-0 pr-3 flex items-center pointer-events-none">
                                    <span id="discount_suffix" class="text-gray-500">
                                        @if($deal->type == 'percentage') %
                                        @elseif($deal->type == 'fixed') Ksh
                                        @endif
                                    </span>
                                </div>
                            </div>
                            @error('discount_value')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>
                        
                        <!-- Minimum Purchase -->
                        <div>
                            <label for="min_purchase" class="block text-sm font-medium text-gray-700 mb-2">Minimum Purchase (Optional)</label>
                            <div class="relative">
                                <input type="number" 
                                       name="min_purchase" 
                                       id="min_purchase" 
                                       value="{{ old('min_purchase', $deal->min_purchase) }}"
                                       step="0.01"
                                       min="0"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0A9961] focus:border-[#0A9961] pr-10">
                                <div class="absolute inset-y-0 right-0 pr-3 flex items-center pointer-events-none">
                                    <span class="text-gray-500">Ksh</span>
                                </div>
                            </div>
                            <p class="mt-1 text-xs text-gray-500">Minimum cart total to apply this deal</p>
                            @error('min_purchase')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>
                        
                        <!-- Usage Limit -->
                        <div>
                            <label for="usage_limit" class="block text-sm font-medium text-gray-700 mb-2">Usage Limit (Optional)</label>
                            <input type="number" 
                                   name="usage_limit" 
                                   id="usage_limit" 
                                   value="{{ old('usage_limit', $deal->usage_limit) }}"
                                   min="1"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0A9961] focus:border-[#0A9961]">
                            <p class="mt-1 text-xs text-gray-500">Maximum number of times this deal can be used</p>
                            @error('usage_limit')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>
                    </div>
                </div>
                
                <!-- Date & Time -->
                <div>
                    <h3 class="text-lg font-medium text-gray-900 mb-4">Date & Time</h3>
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <!-- Start Date -->
                        <div>
                            <label for="start_date" class="block text-sm font-medium text-gray-700 mb-2">Start Date *</label>
                            <input type="datetime-local" 
                                   name="start_date" 
                                   id="start_date" 
                                   value="{{ old('start_date', $deal->start_date->format('Y-m-d\TH:i')) }}"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0A9961] focus:border-[#0A9961]"
                                   required>
                            @error('start_date')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>
                        
                        <!-- End Date -->
                        <div>
                            <label for="end_date" class="block text-sm font-medium text-gray-700 mb-2">End Date *</label>
                            <input type="datetime-local" 
                                   name="end_date" 
                                   id="end_date" 
                                   value="{{ old('end_date', $deal->end_date->format('Y-m-d\TH:i')) }}"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0A9961] focus:border-[#0A9961]"
                                   required>
                            @error('end_date')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>
                    </div>
                </div>
                
                <!-- Image Upload -->
                <div>
                    <h3 class="text-lg font-medium text-gray-900 mb-4">Deal Image</h3>
                    <div class="flex items-center space-x-6">
                        <!-- Current Image Preview -->
                        @if($deal->image)
                        <div id="current-image" class="relative">
                            <div class="w-40 h-40 border-2 border-gray-300 rounded-lg overflow-hidden">
                                <img src="{{ asset('storage/' . $deal->image) }}" 
                                     alt="{{ $deal->title }}"
                                     class="w-full h-full object-cover">
                            </div>
                            <div class="mt-2 flex items-center space-x-2">
                                <label class="flex items-center text-sm text-gray-600 cursor-pointer">
                                    <input type="checkbox" 
                                           name="remove_image" 
                                           value="1"
                                           class="h-4 w-4 text-red-600 focus:ring-red-500 border-gray-300 rounded">
                                    <span class="ml-2">Remove image</span>
                                </label>
                            </div>
                        </div>
                        @endif
                        
                        <!-- Upload Area -->
                        <div class="flex-1">
                            <div class="border-2 border-gray-300 border-dashed rounded-lg p-6 text-center">
                                @if($deal->image)
                                <p class="text-sm text-gray-700 mb-4">Upload new image to replace current one</p>
                                @else
                                <div class="mb-4">
                                    <i class="fas fa-cloud-upload-alt text-gray-400 text-3xl"></i>
                                </div>
                                @endif
                                
                                <label for="image" class="cursor-pointer">
                                    <span class="block text-sm font-medium text-gray-700 mb-2">
                                        {{ $deal->image ? 'Upload New Image' : 'Upload Deal Image' }}
                                    </span>
                                    <span class="block text-xs text-gray-500 mb-4">PNG, JPG up to 2MB</span>
                                    <input type="file" 
                                           name="image" 
                                           id="image" 
                                           accept="image/*"
                                           class="hidden"
                                           onchange="previewImage(event)">
                                    <span class="inline-block bg-gray-100 hover:bg-gray-200 text-gray-700 font-medium py-2 px-4 rounded-lg transition duration-200">
                                        Choose File
                                    </span>
                                </label>
                            </div>
                            
                            <!-- New Image Preview -->
                            <div id="new-image-preview" class="hidden mt-4">
                                <p class="text-sm font-medium text-gray-700 mb-2">New Image Preview:</p>
                                <div class="w-40 h-40 border-2 border-gray-300 border-dashed rounded-lg overflow-hidden">
                                    <img id="preview-image" class="w-full h-full object-cover" src="" alt="Preview">
                                </div>
                            </div>
                            
                            @error('image')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>
                    </div>
                </div>
                
                <!-- Products Selection -->
                <div>
                    <h3 class="text-lg font-medium text-gray-900 mb-4">Products in This Deal</h3>
                    <div class="bg-gray-50 rounded-lg p-4">
                        <div class="mb-4">
                            <div class="flex items-center justify-between mb-2">
                                <label class="text-sm font-medium text-gray-700">Select Products</label>
                                <span class="text-xs text-gray-500">
                                    <span id="selected-count">{{ count($selectedProducts) }}</span> products selected
                                </span>
                            </div>
                            
                            <!-- Search Products -->
                            <div class="relative mb-4">
                                <input type="text" 
                                       id="product-search" 
                                       placeholder="Search products..."
                                       class="w-full px-4 py-2 pl-10 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0A9961] focus:border-[#0A9961]">
                                <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                    <i class="fas fa-search text-gray-400"></i>
                                </div>
                            </div>
                            
                            <!-- Products Grid -->
                            <div class="max-h-60 overflow-y-auto border border-gray-200 rounded-lg bg-white">
                                <div id="products-container" class="p-2 grid grid-cols-1 gap-2">
                                    @foreach($products as $product)
                                    <label class="flex items-center p-2 hover:bg-gray-50 rounded cursor-pointer">
                                        <input type="checkbox" 
                                               name="products[]" 
                                               value="{{ $product->id }}"
                                               {{ in_array($product->id, $selectedProducts) ? 'checked' : '' }}
                                               class="h-4 w-4 text-[#0A9961] focus:ring-[#0A9961] border-gray-300 rounded"
                                               onchange="updateSelectedCount()">
                                        <div class="ml-3 flex items-center flex-1">
                                            @if($product->image)
                                            <img src="{{ $product->image_url }}" 
                                                 alt="{{ $product->name }}"
                                                 class="w-8 h-8 rounded mr-3 object-cover">
                                            @else
                                            <div class="w-8 h-8 rounded bg-gray-100 flex items-center justify-center mr-3">
                                                <i class="fas fa-box text-gray-400 text-xs"></i>
                                            </div>
                                            @endif
                                            <div class="flex-1 min-w-0">
                                                <div class="text-sm font-medium text-gray-900 truncate">{{ $product->name }}</div>
                                                <div class="text-xs text-gray-500">Ksh {{ number_format($product->price) }}</div>
                                            </div>
                                            <div class="text-xs text-gray-500">
                                                Stock: {{ $product->stock }}
                                            </div>
                                        </div>
                                    </label>
                                    @endforeach
                                </div>
                            </div>
                        </div>
                        @error('products')
                            <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                        @enderror
                    </div>
                </div>
                
                <!-- Additional Settings -->
                <div>
                    <h3 class="text-lg font-medium text-gray-900 mb-4">Additional Settings</h3>
                    <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
                        <!-- Sort Order -->
                        <div>
                            <label for="sort_order" class="block text-sm font-medium text-gray-700 mb-2">Sort Order</label>
                            <input type="number" 
                                   name="sort_order" 
                                   id="sort_order" 
                                   value="{{ old('sort_order', $deal->sort_order) }}"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0A9961] focus:border-[#0A9961]">
                            <p class="mt-1 text-xs text-gray-500">Lower numbers appear first</p>
                            @error('sort_order')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>
                        
                        <!-- Status -->
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Status</label>
                            <div class="flex items-center">
                                <input type="checkbox" 
                                       name="is_active" 
                                       id="is_active" 
                                       value="1"
                                       {{ old('is_active', $deal->is_active) ? 'checked' : '' }}
                                       class="h-4 w-4 text-[#0A9961] focus:ring-[#0A9961] border-gray-300 rounded">
                                <label for="is_active" class="ml-2 text-sm text-gray-700">Active</label>
                            </div>
                            <p class="mt-1 text-xs text-gray-500">Deal will be visible to customers</p>
                        </div>
                        
                        <!-- Featured -->
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Featured</label>
                            <div class="flex items-center">
                                <input type="checkbox" 
                                       name="featured" 
                                       id="featured" 
                                       value="1"
                                       {{ old('featured', $deal->featured) ? 'checked' : '' }}
                                       class="h-4 w-4 text-[#0A9961] focus:ring-[#0A9961] border-gray-300 rounded">
                                <label for="featured" class="ml-2 text-sm text-gray-700">Featured Deal</label>
                            </div>
                            <p class="mt-1 text-xs text-gray-500">Show as featured on deals page</p>
                        </div>
                    </div>
                </div>
                
                <!-- Deal Statistics -->
                <div class="bg-gray-50 rounded-lg p-6">
                    <h3 class="text-lg font-medium text-gray-900 mb-4">Deal Statistics</h3>
                    <div class="grid grid-cols-1 md:grid-cols-4 gap-4">
                        <div class="bg-white rounded-lg p-4 text-center border border-gray-200">
                            <div class="text-2xl font-bold text-gray-900">{{ $deal->products->count() }}</div>
                            <div class="text-sm text-gray-600">Products</div>
                        </div>
                        <div class="bg-white rounded-lg p-4 text-center border border-gray-200">
                            <div class="text-2xl font-bold text-gray-900">{{ $deal->used_count }}</div>
                            <div class="text-sm text-gray-600">Times Used</div>
                        </div>
                        <div class="bg-white rounded-lg p-4 text-center border border-gray-200">
                            <div class="text-2xl font-bold text-gray-900">{{ $deal->usage_limit ?? '∞' }}</div>
                            <div class="text-sm text-gray-600">Usage Limit</div>
                        </div>
                        <div class="bg-white rounded-lg p-4 text-center border border-gray-200">
                            <div class="text-2xl font-bold text-gray-900">
                                @if($deal->end_date > now())
                                    {{ $deal->end_date->diffForHumans(['parts' => 2]) }}
                                @else
                                    Expired
                                @endif
                            </div>
                            <div class="text-sm text-gray-600">Time Left</div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Form Actions -->
            <div class="bg-gray-50 px-6 py-4 border-t border-gray-200">
                <div class="flex justify-between items-center">
                    <div>
                        <a href="{{ route('deals.show', $deal->slug) }}" 
                           target="_blank"
                           class="text-[#0A9961] hover:text-[#088552] font-medium no-underline flex items-center">
                            <i class="fas fa-external-link-alt mr-2"></i> View Live Deal
                        </a>
                    </div>
                    <div class="flex space-x-3">
                        <a href="{{ route('admin.deals.index') }}" 
                           class="px-4 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50 no-underline transition duration-200">
                            Cancel
                        </a>
                        <button type="submit" 
                                class="bg-[#0A9961] hover:bg-[#088552] text-white font-medium py-2 px-6 rounded-lg transition duration-200">
                            Update Deal
                        </button>
                    </div>
                </div>
            </div>
        </form>
    </div>
</div>

<script>
    // Update discount label based on deal type
    document.getElementById('type').addEventListener('change', function() {
        const type = this.value;
        const label = document.getElementById('discount_label');
        const suffix = document.getElementById('discount_suffix');
        
        if (type === 'percentage') {
            label.textContent = 'Discount Percentage *';
            suffix.textContent = '%';
        } else if (type === 'fixed') {
            label.textContent = 'Discount Amount *';
            suffix.textContent = 'Ksh';
        } else {
            label.textContent = 'Special Value';
            suffix.textContent = '';
        }
    });
    
    // Image preview
    function previewImage(event) {
        const input = event.target;
        const preview = document.getElementById('preview-image');
        const previewContainer = document.getElementById('new-image-preview');
        
        if (input.files && input.files[0]) {
            const reader = new FileReader();
            
            reader.onload = function(e) {
                preview.src = e.target.result;
                previewContainer.classList.remove('hidden');
            }
            
            reader.readAsDataURL(input.files[0]);
        }
    }
    
    // Product search
    document.getElementById('product-search').addEventListener('input', function(e) {
        const searchTerm = e.target.value.toLowerCase();
        const products = document.querySelectorAll('#products-container label');
        
        products.forEach(product => {
            const productName = product.querySelector('.text-sm').textContent.toLowerCase();
            if (productName.includes(searchTerm)) {
                product.style.display = 'flex';
            } else {
                product.style.display = 'none';
            }
        });
    });
    
    // Update selected count
    function updateSelectedCount() {
        const selectedCount = document.querySelectorAll('input[name="products[]"]:checked').length;
        document.getElementById('selected-count').textContent = selectedCount;
    }
    
    // Initialize
    document.addEventListener('DOMContentLoaded', function() {
        updateSelectedCount();
    });
</script>
@endsection