<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class SettingsController extends Controller
{
    /**
     * Display the settings page.
     */
    public function index()
    {
        $user = Auth::user();
        
        // Get system settings using the getAllSettings method
        $systemSettings = \App\Models\SystemSetting::getAllSettings();
        
        return view('admin.settings.index', compact('user', 'systemSettings'));
    }

    /**
     * Update profile information.
     */
    public function updateProfile(Request $request)
    {
        $user = Auth::user();
        
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255|unique:users,email,' . $user->id,
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput()
                ->with('active_tab', 'profile');
        }

        $user->update([
            'name' => $request->name,
            'email' => $request->email,
        ]);

        return redirect()->route('admin.settings')
            ->with('success', 'Profile updated successfully!')
            ->with('active_tab', 'profile');
    }

    /**
     * Update password.
     */
    public function updatePassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'current_password' => 'required',
            'new_password' => 'required|min:8|confirmed',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput()
                ->with('active_tab', 'password');
        }

        $user = Auth::user();

        if (!Hash::check($request->current_password, $user->password)) {
            return redirect()->back()
                ->with('error', 'Current password is incorrect.')
                ->with('active_tab', 'password');
        }

        $user->update([
            'password' => Hash::make($request->new_password),
        ]);

        return redirect()->route('admin.settings')
            ->with('success', 'Password updated successfully!')
            ->with('active_tab', 'password');
    }

    /**
     * Update system settings.
     */
    public function updateSystem(Request $request)
    {
        // DEBUG - Log all incoming form data
        \Log::info("=== System Settings Update ===");
        \Log::info("Full form data:", $request->all());
        
        // Log specific payment method values
        \Log::info("mpesa_enabled value: " . ($request->has('mpesa_enabled') ? 'YES' : 'NO'));
        \Log::info("cash_on_delivery_enabled value: " . ($request->has('cash_on_delivery_enabled') ? 'YES' : 'NO'));
        \Log::info("direct_paybill_enabled value: " . ($request->has('direct_paybill_enabled') ? 'YES' : 'NO'));

        // Validate system settings
        $validator = Validator::make($request->all(), [
            'site_name' => 'nullable|string|max:255',
            'site_email' => 'nullable|email|max:255',
            'site_phone' => 'nullable|string|max:20',
            'site_address' => 'nullable|string|max:500',
            'currency' => 'nullable|string|max:10',
            'timezone' => 'nullable|string|max:50',
            'mpesa_enabled' => 'nullable',
            'cash_on_delivery_enabled' => 'nullable',
            'direct_paybill_enabled' => 'nullable',
            'paybill_number' => 'nullable|string|max:20',
            'paybill_account_number' => 'nullable|string|max:20',
            'paybill_instructions' => 'nullable|string|max:500',
        ]);

        if ($validator->fails()) {
            \Log::error("Validation failed:", $validator->errors()->toArray());
            return redirect()->back()
                ->withErrors($validator)
                ->withInput()
                ->with('active_tab', 'system');
        }

        // Prepare all settings for update
        $settings = [
            'site_name' => $request->site_name,
            'site_email' => $request->site_email,
            'site_phone' => $request->site_phone,
            'site_address' => $request->site_address,
            'currency' => $request->currency,
            'timezone' => $request->timezone,
            'mpesa_enabled' => $request->has('mpesa_enabled') ? '1' : '0',
            'cash_on_delivery_enabled' => $request->has('cash_on_delivery_enabled') ? '1' : '0',
            'direct_paybill_enabled' => $request->has('direct_paybill_enabled') ? '1' : '0',
            'paybill_number' => $request->paybill_number,
            'paybill_account_number' => $request->paybill_account_number,
            'paybill_instructions' => $request->paybill_instructions,
        ];

        \Log::info("Prepared settings for save:", $settings);

        // Save all settings using the SystemSetting model
        try {
            \App\Models\SystemSetting::saveSettings($settings);
            \Log::info("Settings saved successfully");
        } catch (\Exception $e) {
            \Log::error("Error saving settings: " . $e->getMessage());
            return redirect()->back()
                ->with('error', 'Error saving settings: ' . $e->getMessage())
                ->with('active_tab', 'system');
        }

        // Verify the save
        $savedSettings = \App\Models\SystemSetting::getAllSettings();
        \Log::info("Verification - Saved settings:", [
            'mpesa_enabled' => $savedSettings['mpesa_enabled'] ?? 'NOT FOUND',
            'cash_on_delivery_enabled' => $savedSettings['cash_on_delivery_enabled'] ?? 'NOT FOUND',
            'direct_paybill_enabled' => $savedSettings['direct_paybill_enabled'] ?? 'NOT FOUND',
            'paybill_number' => $savedSettings['paybill_number'] ?? 'NOT FOUND',
        ]);

        return redirect()->route('admin.settings')
            ->with('success', 'System settings updated successfully!')
            ->with('active_tab', 'system');
    }
}