<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Services\MpesaService;

class TestMpesaService extends Command
{
    protected $signature = 'test:mpesa';
    protected $description = 'Test M-Pesa Service';

    public function handle()
    {
        $mpesa = new MpesaService();
        
        $this->info('Testing M-Pesa Service...');
        
        // Test 1: Get Access Token
        $this->info('1. Getting access token...');
        $token = $mpesa->getAccessToken();
        
        if ($token) {
            $this->info('   ✓ Access token obtained');
            $this->line('   Token: ' . substr($token, 0, 20) . '...');
        } else {
            $this->error('   ✗ Failed to get access token');
            return 1;
        }
        
        // Test 2: Test STK Push (optional - comment out if you don't want to test actual payment)
        $this->info('2. Testing STK Push (simulated)...');
        
        // We'll create a simple phone formatting test since the method is private
        $this->info('3. Testing phone formatting logic...');
        
        $testCases = [
            ['input' => '0712345678', 'expected' => '254712345678'],
            ['input' => '254712345678', 'expected' => '254712345678'],
            ['input' => '+254712345678', 'expected' => '254712345678'],
            ['input' => '712345678', 'expected' => '254712345678'],
        ];
        
        foreach ($testCases as $case) {
            $phone = $case['input'];
            $expected = $case['expected'];
            
            // Manual formatting (same logic as in MpesaService)
            $phone = preg_replace('/[^0-9]/', '', $phone);
            
            if (substr($phone, 0, 1) == '0') {
                $phone = '254' . substr($phone, 1);
            } elseif (substr($phone, 0, 4) == '+254') {
                $phone = substr($phone, 1);
            } elseif (substr($phone, 0, 3) == '254') {
                // Already correct
            } elseif (strlen($phone) == 9) {
                $phone = '254' . $phone;
            }
            
            if ($phone === $expected) {
                $this->info("   ✓ {$case['input']} → {$phone}");
            } else {
                $this->error("   ✗ {$case['input']} → {$phone} (expected: {$expected})");
            }
        }
        
        $this->info('4. Service test completed successfully!');
        $this->newLine();
        $this->info('Next steps:');
        $this->line('   1. Visit /test-mpesa to test payment');
        $this->line('   2. Visit /checkout to test full checkout flow');
        
        return 0;
    }
}