@extends('layouts.app')

@section('title', 'All Products - Finepoint Electronics')

@section('content')
<div class="max-w-7xl mx-auto px-4 py-8">
    <!-- Page Header -->
    <div class="mb-8">
        <div class="flex flex-col md:flex-row md:items-center justify-between gap-4">
            <div class="text-center md:text-left">
                <h1 class="text-2xl font-bold text-gray-900 mb-1">All Products</h1>
                <p class="text-gray-600 text-sm">Browse our premium electronics collection</p>
            </div>
            
            <div class="flex flex-col sm:flex-row items-center gap-3">
                <!-- Search Bar -->
                <div class="relative w-full sm:w-64">
                    <div class="relative">
                        <input type="text" 
                               id="searchInput"
                               placeholder="Search products..." 
                               class="w-full pl-10 pr-10 py-2.5 border border-gray-300 rounded-lg text-sm focus:outline-none focus:ring-2 focus:ring-[#0A9961] focus:border-transparent transition duration-200"
                               onkeyup="debouncedSearch()">
                        <div class="absolute left-3 top-3 text-gray-400">
                            <i class="fas fa-search text-sm"></i>
                        </div>
                        <!-- Clear search button -->
                        <button type="button" 
                                id="clearSearch"
                                class="absolute right-9 top-3 text-gray-400 hover:text-gray-600 hidden"
                                onclick="clearSearch()">
                            <i class="fas fa-times text-sm"></i>
                        </button>
                        <!-- Filter toggle button -->
                        <button type="button" 
                                id="filterToggle"
                                class="absolute right-2.5 top-2.5 text-[#0A9961] hover:text-[#088552]"
                                onclick="toggleFilters()">
                            <i class="fas fa-filter text-sm"></i>
                        </button>
                    </div>
                    
                    <!-- Filters Panel -->
                    <div id="filtersPanel" class="hidden absolute z-10 w-full mt-2 p-4 bg-white rounded-lg shadow-lg border border-gray-200">
                        <div class="mb-4">
                            <h4 class="font-medium text-gray-900 text-sm mb-2">Filter by Category</h4>
                            <select id="categoryFilter" 
                                    class="w-full px-3 py-2 border border-gray-300 rounded text-sm focus:outline-none focus:ring-1 focus:ring-[#0A9961]"
                                    onchange="applyFilters()">
                                <option value="">All Categories</option>
                                @foreach($categories as $category)
                                    <option value="{{ $category->id }}">{{ $category->name }}</option>
                                @endforeach
                            </select>
                        </div>
                        <div class="flex justify-between">
                            <button type="button" 
                                    onclick="clearFilters()"
                                    class="px-3 py-1.5 text-sm text-gray-600 hover:text-gray-800">
                                Clear All
                            </button>
                            <button type="button" 
                                    onclick="toggleFilters()"
                                    class="px-3 py-1.5 bg-[#0A9961] hover:bg-[#088552] text-white text-sm rounded">
                                Apply
                            </button>
                        </div>
                    </div>
                </div>
                
                <!-- Product Count -->
                <div class="text-xs text-gray-500 whitespace-nowrap">
                    <span class="font-medium">{{ $products->total() }}</span> products
                </div>
            </div>
        </div>
    </div>

    <!-- Products Grid -->
    @if($products->count() > 0)
        <div class="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 lg:grid-cols-5 xl:grid-cols-6 gap-4">
            @foreach($products as $product)
            <div class="bg-white border border-gray-200 rounded-lg p-3 product-card hover:shadow-lg transition-all duration-200">
                <!-- Product Image -->
                <a href="{{ route('products.public.show', $product) }}" class="block mb-2 no-underline">
                    @if($product->image)
                        <div class="relative w-full h-32 bg-gray-50 rounded overflow-hidden">
                            <img src="{{ asset($product->image) }}" 
                                 alt="{{ $product->name }}"
                                 class="w-full h-full object-contain p-1 group-hover:scale-105 transition-transform duration-200">
                        </div>
                    @else
                        <div class="w-full h-32 bg-gray-100 rounded flex items-center justify-center">
                            <i class="fas fa-box text-gray-300 text-xl"></i>
                        </div>
                    @endif
                </a>

                <!-- Product Info -->
                <div class="p-2">
                    <!-- Product Name -->
                    <h3 class="font-medium text-gray-900 text-sm mb-1 line-clamp-2 h-10">
                        <a href="{{ route('products.public.show', $product) }}" class="hover:text-[#0A9961] no-underline">
                            {{ $product->name }}
                        </a>
                    </h3>

                    <!-- Brand -->
                    @if($product->brand)
                    <div class="mb-2">
                        <span class="text-xs text-gray-500 truncate block">{{ $product->brand }}</span>
                    </div>
                    @endif

                    <!-- Price -->
                    <div class="mb-2">
                        <span class="text-base font-bold text-[#0A9961]">
                            Ksh {{ number_format($product->price) }}
                        </span>
                    </div>

                    <!-- Stock & Action -->
                    <div class="flex items-center justify-between">
                        <!-- Stock Status -->
                        @if($product->stock > 0)
                            @if($product->stock <= 10)
                                <span class="text-xs text-yellow-600 bg-yellow-50 px-2 py-1 rounded">Low Stock</span>
                            @else
                                <span class="text-xs text-green-600 bg-green-50 px-2 py-1 rounded">In Stock</span>
                            @endif
                        @else
                            <span class="text-xs text-red-600 bg-red-50 px-2 py-1 rounded">Out of Stock</span>
                        @endif
                        
                        <!-- Add to Cart Button -->
                        <button onclick="addToCart(event, {{ $product->id }}, '{{ addslashes($product->name) }}', {{ $product->price }}, {{ $product->stock }})" 
                                class="text-[#0A9961] hover:text-[#088552] p-1 {{ $product->stock <= 0 ? 'opacity-50 cursor-not-allowed' : '' }} bg-transparent border-none cursor-pointer transition-colors duration-200"
                                {{ $product->stock <= 0 ? 'disabled' : '' }}>
                            <i class="fas fa-cart-plus text-sm"></i>
                        </button>
                    </div>
                </div>
            </div>
            @endforeach
        </div>

        <!-- Pagination -->
        @if($products->hasPages())
        <div class="mt-8">
            {{ $products->links() }}
        </div>
        @endif
    @else
        <!-- Empty State -->
        <div class="text-center py-12 bg-white rounded-xl shadow-sm">
            <div class="w-20 h-20 rounded-full bg-gray-100 flex items-center justify-center mx-auto mb-4">
                <i class="fas fa-box-open text-gray-300 text-2xl"></i>
            </div>
            <h3 class="text-lg font-medium text-gray-900 mb-2">No products found</h3>
            <p class="text-gray-600 mb-6 text-sm">Check back soon for new products.</p>
            <a href="{{ route('home') }}" class="inline-block px-4 py-2 bg-[#0A9961] hover:bg-[#088552] text-white rounded text-sm transition duration-200 no-underline">
                Return to Home
            </a>
        </div>
    @endif
</div>
@endsection

@push('styles')
<style>
    .product-card { 
        transition: all 0.2s ease-in-out; 
    }
    
    .product-card:hover { 
        transform: translateY(-2px); 
        box-shadow: 0 10px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04); 
    }
    
    .line-clamp-2 { 
        display: -webkit-box; 
        -webkit-line-clamp: 2; 
        -webkit-box-orient: vertical; 
        overflow: hidden; 
    }
    
    .h-10 {
        height: 2.5rem;
    }
    
    /* Ensure consistent image sizing */
    .object-contain {
        object-fit: contain;
    }

    /* Clear search button visibility */
    #clearSearch {
        transition: opacity 0.2s ease;
    }
    
    #clearSearch:hover {
        opacity: 0.7;
    }
</style>
@endpush

@push('scripts')
<script>
    // Ensure addToCart function is available globally
    window.addToCart = function(event, productId, productName, price, stock) {
        event.preventDefault();
        event.stopPropagation();
        
        // Check if product is in stock
        if (stock <= 0) {
            showNotification('This product is out of stock', 'error');
            return;
        }
        
        // Show loading state on the button
        const button = event.target.closest('button');
        const originalHTML = button.innerHTML;
        button.innerHTML = '<i class="fas fa-spinner fa-spin text-sm"></i>';
        button.disabled = true;
        
        // Get CSRF token from meta tag
        const csrfToken = document.querySelector('meta[name="csrf-token"]').getAttribute('content');
        
        // Prepare data for the request
        const data = {
            product_id: productId,
            quantity: 1,
            _token: csrfToken
        };
        
        // Make actual API call
        fetch('{{ route("cart.add") }}', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json',
                'X-CSRF-TOKEN': csrfToken
            },
            body: JSON.stringify(data)
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showNotification('✓ Added to cart: ' + productName);
                updateCartCount(data.cart_count);
            } else {
                showNotification(data.message || 'Failed to add to cart', 'error');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showNotification('An error occurred. Please try again.', 'error');
        })
        .finally(() => {
            // Restore button state
            button.innerHTML = originalHTML;
            button.disabled = false;
        });
    };
    
    // Function to show notifications
    window.showNotification = function(message, type = 'success') {
        // Create notification element
        const notification = document.createElement('div');
        const bgColor = type === 'success' ? 'bg-green-500' : 'bg-red-500';
        const icon = type === 'success' ? 'fa-check-circle' : 'fa-exclamation-circle';
        
        notification.className = `${bgColor} text-white px-4 py-3 rounded-lg shadow-lg z-50 fixed top-4 right-4 transition-all duration-300 transform translate-x-full`;
        notification.innerHTML = `
            <div class="flex items-center">
                <i class="fas ${icon} mr-3"></i>
                <span>${message}</span>
                <button onclick="this.parentElement.parentElement.remove()" class="ml-4 text-white hover:text-gray-200">
                    <i class="fas fa-times"></i>
                </button>
            </div>
        `;
        
        // Add to document
        document.body.appendChild(notification);
        
        // Animate in
        setTimeout(() => {
            notification.classList.remove('translate-x-full');
            notification.classList.add('translate-x-0');
        }, 10);
        
        // Remove after 4 seconds
        setTimeout(() => {
            notification.classList.remove('translate-x-0');
            notification.classList.add('translate-x-full');
            setTimeout(() => {
                if (notification.parentNode) {
                    notification.parentNode.removeChild(notification);
                }
            }, 300);
        }, 4000);
    };
    
    // Function to update cart count (from layout)
    window.updateCartCount = function(count = null) {
        if (count !== null) {
            // If count is provided, update directly
            setCartCount(count);
        } else {
            // Fetch current cart count from server
            fetch('{{ route("cart.count") }}')
                .then(response => response.json())
                .then(data => {
                    setCartCount(data.cart_count);
                })
                .catch(error => {
                    console.error('Error fetching cart count:', error);
                });
        }
    };
    
    // Helper function to set cart count in the UI
    function setCartCount(count) {
        const cartCountElement = document.getElementById('cart-count');
        if (cartCountElement) {
            cartCountElement.textContent = count;
        }
    }
    
    // Initialize cart count on page load
    document.addEventListener('DOMContentLoaded', function() {
        updateCartCount();
    });

    // Search and filter functions
    let debounceTimer;
    
    function debouncedSearch() {
        clearTimeout(debounceTimer);
        debounceTimer = setTimeout(() => {
            performSearch();
        }, 300); // 300ms delay
        
        // Show/hide clear button
        const searchInput = document.getElementById('searchInput');
        const clearButton = document.getElementById('clearSearch');
        clearButton.classList.toggle('hidden', searchInput.value === '');
    }

    function performSearch() {
        const searchTerm = document.getElementById('searchInput').value;
        const categoryId = document.getElementById('categoryFilter').value;
        
        // Update URL with search parameters
        let url = new URL(window.location.href);
        let params = new URLSearchParams(url.search);
        
        if (searchTerm) {
            params.set('search', searchTerm);
        } else {
            params.delete('search');
        }
        
        if (categoryId) {
            params.set('category', categoryId);
        } else {
            params.delete('category');
        }
        
        params.delete('page'); // Reset to first page when searching
        
        // Redirect with new params
        window.location.href = url.pathname + '?' + params.toString();
    }

    function clearSearch() {
        document.getElementById('searchInput').value = '';
        document.getElementById('clearSearch').classList.add('hidden');
        performSearch();
    }

    function toggleFilters() {
        const filtersPanel = document.getElementById('filtersPanel');
        filtersPanel.classList.toggle('hidden');
    }

    function applyFilters() {
        performSearch();
        toggleFilters(); // Close filters panel
    }

    function clearFilters() {
        document.getElementById('searchInput').value = '';
        document.getElementById('categoryFilter').value = '';
        document.getElementById('clearSearch').classList.add('hidden');
        performSearch();
    }

    // Close filters when clicking outside
    document.addEventListener('click', function(event) {
        const filtersPanel = document.getElementById('filtersPanel');
        const filterToggle = document.getElementById('filterToggle');
        
        if (filtersPanel && !filtersPanel.classList.contains('hidden')) {
            if (!filtersPanel.contains(event.target) && !filterToggle.contains(event.target)) {
                filtersPanel.classList.add('hidden');
            }
        }
    });

    // Load current search params on page load
    document.addEventListener('DOMContentLoaded', function() {
        const urlParams = new URLSearchParams(window.location.search);
        
        // Set search input
        const searchParam = urlParams.get('search');
        if (searchParam) {
            document.getElementById('searchInput').value = searchParam;
            document.getElementById('clearSearch').classList.remove('hidden');
        }
        
        // Set category filter
        const categoryParam = urlParams.get('category');
        if (categoryParam) {
            document.getElementById('categoryFilter').value = categoryParam;
        }
    });
</script>
@endpush