<!DOCTYPE html>
<html lang="{{ str_replace('_', '-', app()->getLocale()) }}">
<head>
    <script src="https://cdn.tailwindcss.com"></script>
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');
        body { font-family: 'Inter', sans-serif; }
        .product-card { transition: all 0.2s ease-in-out; }
        .product-card:hover { transform: translateY(-2px); box-shadow: 0 10px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04); }
        .line-clamp-2 { display: -webkit-box; -webkit-line-clamp: 2; -webkit-box-orient: vertical; overflow: hidden; }
        
        /* Fix for links */
        a { cursor: pointer; }
    </style>

    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <meta name="csrf-token" content="{{ csrf_token() }}">

    <title>{{ $category->name }} - Finepoint Electronics</title>
    
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
<script>
    // Function to show notifications
    function showNotification(message, type = 'success') {
        // Create notification element
        const notification = document.createElement('div');
        const bgColor = type === 'success' ? 'bg-green-500' : 'bg-red-500';
        const icon = type === 'success' ? 'fa-check-circle' : 'fa-exclamation-circle';
        
        notification.className = `${bgColor} text-white px-4 py-3 rounded-lg shadow-lg z-50 fixed top-4 right-4 transition-all duration-300 transform translate-x-full`;
        notification.innerHTML = `
            <div class="flex items-center">
                <i class="fas ${icon} mr-3"></i>
                <span>${message}</span>
                <button onclick="this.parentElement.parentElement.remove()" class="ml-4 text-white hover:text-gray-200">
                    <i class="fas fa-times"></i>
                </button>
            </div>
        `;
        
        // Add to document
        document.body.appendChild(notification);
        
        // Animate in
        setTimeout(() => {
            notification.classList.remove('translate-x-full');
            notification.classList.add('translate-x-0');
        }, 10);
        
        // Remove after 4 seconds
        setTimeout(() => {
            notification.classList.remove('translate-x-0');
            notification.classList.add('translate-x-full');
            setTimeout(() => {
                if (notification.parentNode) {
                    notification.parentNode.removeChild(notification);
                }
            }, 300);
        }, 4000);
    }
    
    // Main function to add product to cart
    function addToCart(event, productId, productName, price, stock) {
        event.preventDefault(); // Prevent any default behavior
        event.stopPropagation(); // Stop event bubbling
        
        // Check if product is in stock
        if (stock <= 0) {
            showNotification('This product is out of stock', 'error');
            return;
        }
        
        // Show loading state on the button
        const button = event.target.closest('button');
        const originalHTML = button.innerHTML;
        button.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
        button.disabled = true;
        
        // Get CSRF token from meta tag
        const csrfToken = document.querySelector('meta[name="csrf-token"]').getAttribute('content');
        
        // Prepare data for the request
        const data = {
            product_id: productId,
            quantity: 1,
            _token: csrfToken
        };
        
        // Make actual API call
        fetch('{{ route("cart.add") }}', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json',
                'X-CSRF-TOKEN': csrfToken
            },
            body: JSON.stringify(data)
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showNotification('✓ Added to cart: ' + productName);
                updateCartCount(data.cart_count);
            } else {
                showNotification(data.message || 'Failed to add to cart', 'error');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showNotification('An error occurred. Please try again.', 'error');
        })
        .finally(() => {
            // Restore button state
            button.innerHTML = originalHTML;
            button.disabled = false;
        });
    }
    
    // Function to update cart count
    function updateCartCount(count = null) {
        if (count !== null) {
            // If count is provided, update directly
            setCartCount(count);
        } else {
            // Fetch current cart count from server
            fetch('{{ route("cart.count") }}')
                .then(response => response.json())
                .then(data => {
                    setCartCount(data.cart_count);
                })
                .catch(error => {
                    console.error('Error fetching cart count:', error);
                });
        }
    }
    
    // Helper function to set cart count in the UI
    function setCartCount(count) {
        const cartCountElement = document.getElementById('cart-count');
        if (cartCountElement) {
            cartCountElement.textContent = count;
        }
    }
    
    // Initialize cart count on page load
    document.addEventListener('DOMContentLoaded', function() {
        updateCartCount();
        
        // Add event listeners to all product links to prevent cart button interference
        document.querySelectorAll('.product-card a').forEach(link => {
            link.addEventListener('click', function(e) {
                // Allow normal navigation for product links
                return true;
            });
        });
    });
</script>
</head>
<body class="bg-gray-50">
    <!-- Navigation (Same as welcome page) -->
    <nav class="bg-white shadow-sm sticky top-0 z-50">
        <div class="max-w-7xl mx-auto px-4">
            <div class="flex justify-between h-14 items-center">
                <!-- Logo -->
                <a href="{{ route('home') }}" class="flex items-center no-underline">
                    <div class="h-8 w-8 rounded-md bg-[#0A9961] flex items-center justify-center mr-2">
                        <i class="fas fa-microchip text-white text-sm"></i>
                    </div>
                    <span class="text-lg font-bold text-gray-900">Finepoint</span>
                    <span class="text-lg font-bold text-[#0A9961]">Electronics</span>
                </a>
                
                <!-- Navigation Links -->
                <div class="hidden md:flex items-center space-x-6">
                    <a href="{{ route('home') }}" class="text-gray-600 hover:text-[#0A9961] text-sm no-underline">Home</a>
                    <a href="{{ route('products.public') }}" class="text-gray-600 hover:text-[#0A9961] text-sm no-underline">Products</a>
                    
                    <!-- Categories Dropdown -->
                    <div class="relative group">
                        <button class="text-gray-600 hover:text-[#0A9961] text-sm no-underline flex items-center focus:outline-none bg-transparent border-none">
                            Categories
                            <i class="fas fa-chevron-down ml-1 text-xs"></i>
                        </button>
                        
                        <!-- Dropdown Menu -->
                        <div class="absolute left-0 mt-2 w-56 bg-white rounded-md shadow-lg border border-gray-200 z-50 opacity-0 invisible group-hover:opacity-100 group-hover:visible transition-all duration-200">
                            <div class="py-2">
                                <div class="px-4 py-2 border-b border-gray-100">
                                    <h3 class="font-medium text-gray-900 text-sm">Browse Categories</h3>
                                </div>
                                
                                @php
                                    // Get categories for dropdown (same as welcome page)
                                    $dropdownCategories = \App\Models\Category::where('is_active', true)
                                        ->whereNull('parent_id')
                                        ->withCount(['products' => function ($query) {
                                            $query->where('is_active', true);
                                        }])
                                        ->get();
                                @endphp
                                
                                @if($dropdownCategories->count() > 0)
                                    @foreach($dropdownCategories as $cat)
                                    <a href="{{ route('category.products', $cat) }}" 
                                       class="block px-4 py-2 text-sm text-gray-700 hover:bg-gray-50 hover:text-[#0A9961] no-underline">
                                        <div class="flex items-center">
                                            @if($cat->image)
                                                <img src="{{ $cat->image_url }}" 
                                                     alt="{{ $cat->name }}"
                                                     class="w-6 h-6 rounded mr-2 object-cover">
                                            @else
                                                <div class="w-6 h-6 rounded bg-gray-100 flex items-center justify-center mr-2">
                                                    <i class="fas fa-folder text-gray-500 text-xs"></i>
                                                </div>
                                            @endif
                                            <span>{{ $cat->name }}</span>
                                            @if($cat->products_count > 0)
                                                <span class="ml-auto text-xs text-gray-500">{{ $cat->products_count }}</span>
                                            @endif
                                        </div>
                                    </a>
                                    @endforeach
                                @else
                                    <div class="px-4 py-3 text-sm text-gray-500 text-center">
                                        No categories yet
                                    </div>
                                @endif
                            </div>
                        </div>
                    </div>
                    
                    <a href="{{ route('home') }}" class="text-gray-600 hover:text-[#0A9961] text-sm no-underline">Deals</a>
                </div>
                
                <!-- Actions -->
                <div class="flex items-center space-x-3">
                    <a href="{{ route('products.public') }}" class="p-2 text-gray-500 hover:text-[#0A9961] no-underline">
                        <i class="fas fa-search"></i>
                    </a>
                    <a href="{{ route('cart.view') }}" class="p-2 text-gray-500 hover:text-[#0A9961] relative no-underline">
                        <i class="fas fa-shopping-cart"></i>
                        <span id="cart-count" class="absolute -top-1 -right-1 bg-[#0A9961] text-white text-xs rounded-full w-4 h-4 flex items-center justify-center">0</span>
                    </a>
                    
                    @auth
                        @if(auth()->user()->isAdmin())
                            <a href="{{ route('admin.dashboard') }}" class="bg-[#0A9961] hover:bg-[#088552] text-white px-3 py-1.5 rounded text-sm no-underline">
                                Admin
                            </a>
                        @else
                            <form method="POST" action="{{ route('logout') }}" class="inline">
                                @csrf
                                <button type="submit" class="text-gray-600 hover:text-[#0A9961] text-sm bg-transparent border-none cursor-pointer">
                                    Logout
                                </button>
                            </form>
                        @endif
                    @else
                        <a href="{{ route('login') }}" class="text-gray-600 hover:text-[#0A9961] text-sm no-underline">
                            Login
                        </a>
                    @endauth
                </div>
            </div>
        </div>
    </nav>

    <!-- Category Header -->
    <div class="bg-white border-b border-gray-200">
        <div class="max-w-7xl mx-auto px-4 py-8">
            <div class="flex flex-col md:flex-row md:items-center justify-between">
                <div class="flex items-center mb-4 md:mb-0">
                    @if($category->image)
                        <img src="{{ $category->image_url }}"  
                             alt="{{ $category->name }}"
                             class="w-16 h-16 rounded-lg object-cover mr-4">
                    @else
                        <div class="w-16 h-16 rounded-lg bg-gray-100 flex items-center justify-center mr-4">
                            <i class="fas fa-folder text-gray-600 text-2xl"></i>
                        </div>
                    @endif
                    <div>
                        <h1 class="text-2xl font-bold text-gray-900">{{ $category->name }}</h1>
                        <p class="text-gray-600 mt-1">
                            {{ $products->total() }} {{ $products->total() == 1 ? 'product' : 'products' }} available
                        </p>
                    </div>
                </div>
                
                <div class="flex space-x-3">
                    <a href="{{ route('products.public') }}" 
                       class="inline-flex items-center border border-gray-300 hover:border-[#0A9961] text-gray-700 hover:text-[#0A9961] font-medium py-2 px-4 rounded-lg transition duration-200 no-underline">
                        <i class="fas fa-boxes mr-2"></i>All Products
                    </a>
                    <a href="{{ route('home') }}" 
                       class="inline-flex items-center border border-gray-300 hover:border-[#0A9961] text-gray-700 hover:text-[#0A9961] font-medium py-2 px-4 rounded-lg transition duration-200 no-underline">
                        <i class="fas fa-home mr-2"></i>Back Home
                    </a>
                </div>
            </div>
            
            @if($category->description)
            <div class="mt-6 p-4 bg-gray-50 rounded-lg">
                <p class="text-gray-700">{{ $category->description }}</p>
            </div>
            @endif
        </div>
    </div>

    <!-- Products Grid -->
    <div class="max-w-7xl mx-auto px-4 py-8">
        @if($products->count() > 0)
        <div class="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 lg:grid-cols-5 xl:grid-cols-6 gap-4">
            @foreach($products as $product)
            <div class="bg-white border border-gray-200 rounded-lg p-3 product-card">
                <!-- Product Image -->
                <a href="{{ route('products.public.show', $product) }}" class="block mb-2 no-underline">
                    @if($product->image)
                        <div class="relative w-full h-32 bg-gray-50 rounded overflow-hidden">
                            <img src="{{ asset($product->image) }}" 
                                 alt="{{ $product->name }}"
                                 class="w-full h-full object-contain p-1">
                        </div>
                    @else
                        <div class="w-full h-32 bg-gray-100 rounded flex items-center justify-center">
                            <i class="fas fa-box text-gray-300 text-xl"></i>
                        </div>
                    @endif
                </a>
                
                <!-- Product Info -->
                <h3 class="font-medium text-gray-900 text-sm mb-1 line-clamp-2 h-10">
                    <a href="{{ route('products.public.show', $product) }}" class="hover:text-[#0A9961] no-underline">
                        {{ $product->name }}
                    </a>
                </h3>
                
                <!-- Brand -->
                @if($product->brand)
                <div class="mb-2">
                    <span class="text-xs text-gray-500 truncate block">{{ $product->brand }}</span>
                </div>
                @endif
                
                <!-- Price -->
                <div class="mb-2">
                    <span class="text-base font-bold text-[#0A9961]">
                        Ksh {{ number_format($product->price) }}
                    </span>
                </div>
                
                <!-- Stock Status Badge -->
                <div class="flex items-center justify-between">
                    @if($product->stock > 0)
                        @if($product->stock <= 10)
                            <span class="text-xs text-yellow-600 bg-yellow-50 px-2 py-1 rounded">Low Stock</span>
                        @else
                            <span class="text-xs text-green-600 bg-green-50 px-2 py-1 rounded">In Stock</span>
                        @endif
                    @else
                        <span class="text-xs text-red-600 bg-red-50 px-2 py-1 rounded">Out of Stock</span>
                    @endif
                    
                    <button onclick="addToCart(event, {{ $product->id }}, '{{ addslashes($product->name) }}', {{ $product->price }}, {{ $product->stock }})" 
                            class="text-[#0A9961] hover:text-[#088552] p-1 {{ $product->stock <= 0 ? 'opacity-50 cursor-not-allowed' : '' }} bg-transparent border-none cursor-pointer"
                            {{ $product->stock <= 0 ? 'disabled' : '' }}>
                        <i class="fas fa-cart-plus"></i>
                    </button>
                </div>
            </div>
            @endforeach
        </div>
        
        <!-- Pagination -->
        @if($products->hasPages())
        <div class="mt-8">
            {{ $products->links() }}
        </div>
        @endif
        
        @else
        <!-- Empty State -->
        <div class="text-center py-16">
            <div class="w-32 h-32 rounded-full bg-gray-100 flex items-center justify-center mx-auto mb-6">
                <i class="fas fa-box-open text-gray-400 text-5xl"></i>
            </div>
            <h3 class="text-xl font-semibold text-gray-900 mb-2">No products in this category</h3>
            <p class="text-gray-600 mb-6">There are no products available in the "{{ $category->name }}" category yet.</p>
            
            <div class="flex justify-center space-x-3">
                <a href="{{ route('products.public') }}" 
                   class="inline-flex items-center bg-[#0A9961] hover:bg-[#088552] text-white font-medium py-3 px-6 rounded-lg transition duration-200 no-underline">
                    <i class="fas fa-boxes mr-2"></i>Browse All Products
                </a>
                <a href="{{ route('home') }}" 
                   class="inline-flex items-center border border-[#0A9961] text-[#0A9961] hover:bg-[#0A9961] hover:text-white font-medium py-3 px-6 rounded-lg transition duration-200 no-underline">
                    <i class="fas fa-home mr-2"></i>Back to Home
                </a>
            </div>
        </div>
        @endif
    </div>

    <!-- Footer -->
    <footer class="bg-gray-900 text-white mt-12">
        <div class="max-w-7xl mx-auto px-4 py-6">
            <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
                <div>
                    <div class="flex items-center mb-3">
                        <div class="h-6 w-6 rounded-md bg-[#0A9961] flex items-center justify-center mr-2">
                            <i class="fas fa-microchip text-white text-xs"></i>
                        </div>
                        <span class="font-bold">Finepoint</span>
                        <span class="font-bold text-[#0A9961]">Electronics</span>
                    </div>
                    <p class="text-gray-400 text-xs">Your trusted electronics store for quality products.</p>
                </div>
                
                <div>
                    <h3 class="font-medium mb-2 text-sm">Quick Links</h3>
                    <ul class="space-y-1 text-xs text-gray-400">
                        <li><a href="{{ route('home') }}" class="hover:text-white no-underline">Home</a></li>
                        <li><a href="{{ route('products.public') }}" class="hover:text-white no-underline">Products</a></li>
                        <li><a href="{{ route('categories.index') }}" class="hover:text-white no-underline">Categories</a></li>
                        <li><a href="{{ route('cart.view') }}" class="hover:text-white no-underline">Cart</a></li>
                    </ul>
                </div>
                
                <div>
                    <h3 class="font-medium mb-2 text-sm">Contact</h3>
                    <ul class="space-y-1 text-xs text-gray-400">
                        <li class="flex items-center">
                            <i class="fas fa-phone mr-2 text-[#0A9961] text-xs"></i>
                            +254700 123 456
                        </li>
                        <li class="flex items-center">
                            <i class="fas fa-envelope mr-2 text-[#0A9961] text-xs"></i>
                            support@finepoint.com
                        </li>
                    </ul>
                </div>
            </div>
            
            <div class="border-t border-gray-800 mt-6 pt-6 text-center text-gray-400 text-xs">
                <p>&copy; {{ date('Y') }} Finepoint Electronics. All rights reserved.</p>
            </div>
        </div>
    </footer>
</body>
</html>