@extends('layouts.admin-layout')

@section('content')
<div class="max-w-4xl mx-auto">
    <!-- Header -->
    <div class="mb-6">
        <div class="flex items-center justify-between">
            <div>
                <h1 class="text-2xl font-bold text-gray-900">Edit Category</h1>
                <p class="text-gray-600 mt-2">Update category information</p>
            </div>
            <div class="flex items-center space-x-2 text-sm text-gray-500">
                <i class="fas fa-calendar"></i>
                <span>Created: {{ $category->created_at->format('M d, Y') }}</span>
                <span class="mx-2">•</span>
                <i class="fas fa-box"></i>
                <span>{{ $category->products_count ?? 0 }} products</span>
            </div>
        </div>
    </div>

    <!-- Form Card -->
    <div class="bg-white rounded-xl shadow-lg overflow-hidden border border-gray-100">
      
      <!-- Card Header -->
      <div class="px-6 py-4 bg-gradient-to-r from-[#0A9961] to-[#10B981]">
        <div class="flex items-center justify-between">
          <div class="flex items-center">
            <div class="p-3 rounded-lg bg-white/20 backdrop-blur-sm mr-4">
              <i class="fas fa-folder-open text-white text-xl"></i>
            </div>
            <div>
              <h2 class="text-lg font-semibold text-white">{{ $category->name }}</h2>
              <p class="text-green-100 text-sm">Update category details</p>
            </div>
          </div>
          <div class="flex items-center space-x-2">
            @if($category->is_active)
              <span class="px-3 py-1 bg-white/20 text-white text-xs font-medium rounded-full">
                <i class="fas fa-circle text-xs mr-1"></i> Active
              </span>
            @else
              <span class="px-3 py-1 bg-white/20 text-white text-xs font-medium rounded-full">
                <i class="fas fa-circle text-xs mr-1"></i> Inactive
              </span>
            @endif
          </div>
        </div>
      </div>
      
      <!-- Form Content -->
      <div class="p-6">
        @if($errors->any())
        <div class="mb-6 p-4 bg-red-50 border-l-4 border-red-500 rounded-r-lg shadow-sm">
          <div class="flex items-center">
            <i class="fas fa-exclamation-triangle text-red-500 mr-3"></i>
            <div class="flex-1">
              <p class="font-medium text-red-800">Please fix the following errors:</p>
              <ul class="mt-1 text-sm text-red-700 list-disc list-inside">
                @foreach($errors->all() as $error)
                  <li>{{ $error }}</li>
                @endforeach
              </ul>
            </div>
          </div>
        </div>
        @endif

        <form method="POST" action="{{ route('admin.categories.update', $category) }}" enctype="multipart/form-data">
          @csrf
          @method('PUT')

          <!-- Category Name -->
          <div class="mb-6">
            <label for="name" class="block text-sm font-medium text-gray-700 mb-2">
              Category Name *
            </label>
            <div class="relative">
              <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                <i class="fas fa-tag text-gray-400"></i>
              </div>
              <input type="text" id="name" name="name" value="{{ old('name', $category->name) }}" required
                     class="w-full pl-10 pr-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[#0A9961] focus:border-transparent transition duration-200"
                     placeholder="Enter category name">
            </div>
          </div>

          <!-- Slug -->
          <div class="mb-6">
            <label for="slug" class="block text-sm font-medium text-gray-700 mb-2">
              URL Slug *
            </label>
            <div class="relative">
              <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                <i class="fas fa-link text-gray-400"></i>
              </div>
              <input type="text" id="slug" name="slug" value="{{ old('slug', $category->slug) }}" required
                     class="w-full pl-10 pr-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[#0A9961] focus:border-transparent transition duration-200"
                     placeholder="category-url-slug">
            </div>
            <p class="mt-1 text-xs text-gray-500">Used in category URLs. Changing this may affect SEO.</p>
          </div>

          <!-- Parent Category -->
          <div class="mb-6">
            <label for="parent_id" class="block text-sm font-medium text-gray-700 mb-2">
              Parent Category
            </label>
            <select id="parent_id" name="parent_id"
                    class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[#0A9961] focus:border-transparent transition duration-200">
              <option value="">No Parent (Top Level)</option>
              @foreach($parentCategories as $parent)
                @if($parent->id !== $category->id)
                  <option value="{{ $parent->id }}" {{ old('parent_id', $category->parent_id) == $parent->id ? 'selected' : '' }}>
                    {{ $parent->name }}
                  </option>
                @endif
              @endforeach
            </select>
            @if($category->children->count() > 0)
              <p class="mt-1 text-xs text-yellow-600">
                <i class="fas fa-exclamation-circle mr-1"></i>
                This category has {{ $category->children->count() }} sub-categories
              </p>
            @endif
          </div>

          <!-- Description -->
          <div class="mb-6">
            <label for="description" class="block text-sm font-medium text-gray-700 mb-2">
              Description
            </label>
            <textarea id="description" name="description" rows="3"
                      class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[#0A9961] focus:border-transparent transition duration-200"
                      placeholder="Enter category description">{{ old('description', $category->description) }}</textarea>
          </div>

          <!-- Image Upload -->
          <div class="mb-6">
            <label for="image" class="block text-sm font-medium text-gray-700 mb-2">
              Category Image
            </label>
            
            <!-- Current Image -->
            @if($category->image)
            <div class="mb-4 p-4 border border-gray-200 rounded-lg bg-gray-50">
              <p class="text-sm font-medium text-gray-700 mb-2">Current Image:</p>
              <div class="flex items-center space-x-4">
                <img src="{{ ($category->image) }}" 
                     alt="{{ $category->name }}" 
                     class="h-20 w-20 rounded-lg object-cover border border-gray-300">
                <div class="flex-1">
                  <p class="text-sm text-gray-600">Upload a new image to replace this one</p>
                  <label class="inline-flex items-center mt-2">
                    <input type="checkbox" name="remove_image" value="1" 
                           class="h-4 w-4 text-red-600 focus:ring-red-500 border-gray-300 rounded">
                    <span class="ml-2 text-sm text-red-600">Remove current image</span>
                  </label>
                </div>
              </div>
            </div>
            @endif
            
            <!-- Upload New Image -->
            <div class="border-2 border-dashed border-gray-300 rounded-lg p-6 text-center hover:border-[#0A9961] transition-colors duration-200">
              <div class="space-y-2">
                <i class="fas fa-image text-3xl text-gray-400 mx-auto"></i>
                <div>
                  <label for="image" class="cursor-pointer inline-flex items-center px-4 py-2 bg-[#0A9961] hover:bg-[#088551] text-white font-medium rounded-lg transition duration-200">
                    <i class="fas fa-upload mr-2"></i>
                    {{ $category->image ? 'Change Image' : 'Upload Image' }}
                  </label>
                  <input type="file" id="image" name="image" class="hidden" accept="image/*">
                </div>
                <p class="text-xs text-gray-500">PNG, JPG, GIF up to 5MB. Recommended: 400x400px</p>
                <p id="fileName" class="text-sm text-gray-900 font-medium">
                  {{ $category->image ? 'Leave empty to keep current image' : 'No file chosen' }}
                </p>
              </div>
            </div>
          </div>

          <!-- Status -->
          <div class="mb-6">
            <label class="block text-sm font-medium text-gray-700 mb-2">
              Status
            </label>
            <div class="flex items-center space-x-4">
              <label class="inline-flex items-center">
                <input type="radio" name="is_active" value="1" {{ old('is_active', $category->is_active) ? 'checked' : '' }}
                       class="h-4 w-4 text-[#0A9961] focus:ring-[#0A9961] border-gray-300">
                <span class="ml-2 text-sm text-gray-700">Active</span>
              </label>
              <label class="inline-flex items-center">
                <input type="radio" name="is_active" value="0" {{ !old('is_active', $category->is_active) ? 'checked' : '' }}
                       class="h-4 w-4 text-gray-600 focus:ring-gray-500 border-gray-300">
                <span class="ml-2 text-sm text-gray-700">Inactive</span>
              </label>
            </div>
          </div>

          <!-- SEO Fields -->
          <div class="mb-6 p-4 bg-gray-50 rounded-lg border border-gray-200">
            <h3 class="text-sm font-medium text-gray-900 mb-3">SEO Settings</h3>
            
            <div class="space-y-4">
              <div>
                <label for="meta_title" class="block text-xs font-medium text-gray-700 mb-1">
                  Meta Title
                </label>
                <input type="text" id="meta_title" name="meta_title" value="{{ old('meta_title', $category->meta_title) }}"
                       class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:outline-none focus:ring-1 focus:ring-[#0A9961] focus:border-transparent"
                       placeholder="Meta title for search engines">
              </div>
              
              <div>
                <label for="meta_description" class="block text-xs font-medium text-gray-700 mb-1">
                  Meta Description
                </label>
                <textarea id="meta_description" name="meta_description" rows="2"
                          class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:outline-none focus:ring-1 focus:ring-[#0A9961] focus:border-transparent"
                          placeholder="Meta description for search engines">{{ old('meta_description', $category->meta_description) }}</textarea>
              </div>
              
              <div>
                <label for="meta_keywords" class="block text-xs font-medium text-gray-700 mb-1">
                  Meta Keywords
                </label>
                <input type="text" id="meta_keywords" name="meta_keywords" value="{{ old('meta_keywords', $category->meta_keywords) }}"
                       class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:outline-none focus:ring-1 focus:ring-[#0A9961] focus:border-transparent"
                       placeholder="keyword1, keyword2, keyword3">
              </div>
            </div>
          </div>

          <!-- Form Actions -->
          <div class="flex items-center justify-between pt-6 border-t border-gray-200">
            <a href="{{ route('admin.categories.index') }}" 
               class="px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition duration-200">
              Cancel
            </a>
            <div class="flex space-x-3">
              <button type="button" 
                      onclick="confirmDelete()"
                      class="px-4 py-2 border border-red-600 text-red-600 rounded-lg hover:bg-red-50 transition duration-200">
                Delete Category
              </button>
              <button type="submit"
                      class="px-6 py-2 bg-[#0A9961] hover:bg-[#088551] text-white font-medium rounded-lg transition duration-200">
                Update Category
              </button>
            </div>
          </div>
        </form>

        <!-- Delete Form -->
        <form id="delete-form" action="{{ route('admin.categories.destroy', $category) }}" method="POST" class="hidden">
          @csrf
          @method('DELETE')
        </form>
      </div>
    </div>
  </div>
</div>

<script>
  // File name display
  document.getElementById('image').addEventListener('change', function(e) {
    const fileName = e.target.files[0]?.name || 'Leave empty to keep current image';
    document.getElementById('fileName').textContent = fileName;
  });
  
  // Auto-generate slug from category name
  document.getElementById('name').addEventListener('input', function(e) {
    const slugInput = document.getElementById('slug');
    const currentSlug = '{{ $category->slug }}';
    
    if (!slugInput.value || slugInput.value === currentSlug) {
      const slug = e.target.value
        .toLowerCase()
        .replace(/[^\w\s-]/g, '')
        .replace(/\s+/g, '-')
        .replace(/--+/g, '-');
      slugInput.value = slug;
    }
  });

  function confirmDelete() {
    if (confirm('Are you sure you want to delete this category? This action cannot be undone.')) {
      document.getElementById('delete-form').submit();
    }
  }
</script>
@endsection