<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class OrderPaidNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public $order;

    /**
     * Create a new notification instance.
     */
    public function __construct($order)
    {
        $this->order = $order;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
                    ->subject('🎉 Order #' . $this->order->order_number . ' Payment Confirmed!')
                    ->greeting('Hello ' . $notifiable->name . '!')
                    ->line('Great news! Payment for your order has been confirmed.')
                    ->line('**Order Details:**')
                    ->line('Order Number: #' . $this->order->order_number)
                    ->line('Amount Paid: Ksh ' . number_format($this->order->total_amount, 2))
                    ->line('Payment Method: ' . $this->order->payment_method)
                    ->line('Date: ' . $this->order->created_at->format('F d, Y h:i A'))
                    ->action('View Order', url('/my-orders/' . $this->order->id))
                    ->line('Your order is now being processed and will be shipped soon.')
                    ->line('Thank you for shopping with Finepoint Electronics!');
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'order_id' => $this->order->id,
            'order_number' => $this->order->order_number,
            'amount' => $this->order->total_amount,
        ];
    }
}