<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

class Deal extends Model
{
    use HasFactory;

    protected $fillable = [
        'title',
        'slug',
        'description',
        'image',
        'type',
        'discount_value',
        'min_purchase',
        'usage_limit',
        'used_count',
        'start_date',
        'end_date',
        'is_active',
        'featured',
        'sort_order'
    ];

    protected $casts = [
        'start_date' => 'datetime',
        'end_date' => 'datetime',
        'is_active' => 'boolean',
        'featured' => 'boolean',
        'discount_value' => 'decimal:2',
        'min_purchase' => 'decimal:2'
    ];

    // Relationship with products
    public function products()
    {
        return $this->belongsToMany(Product::class, 'deal_product');
    }

    // Scope for active deals
    public function scopeActive($query)
    {
        return $query->where('is_active', true)
                    ->where('start_date', '<=', now())
                    ->where('end_date', '>=', now());
    }

    // Scope for featured deals
    public function scopeFeatured($query)
    {
        return $query->where('featured', true)->active();
    }

    // Check if deal is currently active
    public function isActive()
    {
        return $this->is_active && 
               $this->start_date <= now() && 
               $this->end_date >= now();
    }

    // Calculate discounted price for a product
    public function calculateDiscount($originalPrice)
    {
        if (!$this->isActive()) {
            return $originalPrice;
        }

        switch ($this->type) {
            case 'percentage':
                $discount = $originalPrice * ($this->discount_value / 100);
                return max(0, $originalPrice - $discount);
            
            case 'fixed':
                return max(0, $originalPrice - $this->discount_value);
            
            case 'bundle':
            case 'special':
                return $originalPrice; // Special pricing handled differently
        }

        return $originalPrice;
    }

    // Auto-generate slug
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($deal) {
            if (empty($deal->slug)) {
                $deal->slug = Str::slug($deal->title);
            }
        });
    }

    // Get image URL
    public function getImageUrlAttribute()
    {
        if ($this->image) {
            return asset('storage/' . $this->image);
        }
        return asset('images/default-deal.jpg');
    }

    // Check if usage limit is reached
    public function isUsageLimitReached()
    {
        if (is_null($this->usage_limit)) {
            return false;
        }
        return $this->used_count >= $this->usage_limit;
    }

    // Increment usage count
    public function incrementUsage()
    {
        $this->increment('used_count');
    }
}