<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Cart extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'product_id',
        'quantity',
        'original_price', // NEW: Store original price before discount
        'deal_id', // NEW: Reference to the deal
        'deal_type', // NEW: 'percentage', 'fixed', or 'bundle'
        'deal_discount_value', // NEW: The discount amount/percentage
    ];

    protected $casts = [
        'original_price' => 'decimal:2',
        'deal_discount_value' => 'decimal:2',
    ];

    // Relationship with User
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    // Relationship with Product
    public function product()
    {
        return $this->belongsTo(Product::class);
    }

    // NEW: Relationship with Deal
    public function deal()
    {
        return $this->belongsTo(Deal::class);
    }

    // Calculate the actual price considering deals
    public function getActualPriceAttribute()
    {
        // If we have deal information stored, calculate deal price
        if ($this->original_price && $this->deal_type && $this->deal_discount_value) {
            return $this->calculateDealPrice($this->original_price);
        }
        
        // Otherwise use product's current price
        return $this->product->price;
    }

    // Calculate deal price based on deal type
    private function calculateDealPrice($originalPrice)
    {
        switch ($this->deal_type) {
            case 'percentage':
                $discount = ($this->deal_discount_value / 100) * $originalPrice;
                return $originalPrice - $discount;
                
            case 'fixed':
                return max(0, $originalPrice - $this->deal_discount_value);
                
            case 'bundle':
                // For bundle deals, price might be set differently
                // You might want to store bundle price directly
                return $originalPrice; // Or implement bundle logic
                
            default:
                return $originalPrice;
        }
    }

    // Calculate total price for this cart item (using actual price)
    public function getTotalPriceAttribute()
    {
        return $this->quantity * $this->actual_price;
    }

    // Calculate discount amount for display
    public function getDiscountAmountAttribute()
    {
        if ($this->original_price && $this->deal_type && $this->deal_discount_value) {
            $originalTotal = $this->quantity * $this->original_price;
            $dealTotal = $this->quantity * $this->actual_price;
            return $originalTotal - $dealTotal;
        }
        return 0;
    }

    // Check if item has a deal applied
    public function getHasDealAttribute()
    {
        return !empty($this->deal_id) && !empty($this->deal_type);
    }

    // Check if item is in stock
    public function getIsInStockAttribute()
    {
        return $this->quantity <= $this->product->stock;
    }

    // Get formatted deal description
    public function getDealDescriptionAttribute()
    {
        if (!$this->has_deal) return null;
        
        switch ($this->deal_type) {
            case 'percentage':
                return "{$this->deal_discount_value}% OFF";
            case 'fixed':
                return "Ksh " . number_format($this->deal_discount_value) . " OFF";
            case 'bundle':
                return "Bundle Deal";
            default:
                return "Special Offer";
        }
    }

    // Helper method to apply deal to cart item
    public function applyDeal($deal, $originalPrice = null)
    {
        $this->original_price = $originalPrice ?? $this->product->price;
        $this->deal_id = $deal->id;
        $this->deal_type = $deal->type;
        $this->deal_discount_value = $deal->discount_value;
        $this->save();
        
        return $this;
    }

    // Helper method to remove deal from cart item
    public function removeDeal()
    {
        $this->original_price = null;
        $this->deal_id = null;
        $this->deal_type = null;
        $this->deal_discount_value = null;
        $this->save();
        
        return $this;
    }
}