<?php

namespace App\Http\Controllers;

use App\Models\Deal;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use App\Jobs\SendDealNotificationToAllUsers;
use App\Models\User;

class DealsController extends Controller
{
    /**
     * Display the public deals page.
     */
    public function index()
    {
        // Get active deals
        $deals = Deal::active()
                    ->with(['products' => function($query) {
                        $query->active()->inStock();
                    }])
                    ->orderBy('featured', 'desc')
                    ->orderBy('sort_order', 'asc')
                    ->orderBy('created_at', 'desc')
                    ->get();

        // Get featured deals for hero section
        $featuredDeals = Deal::featured()
                            ->with(['products' => function($query) {
                                $query->active()->inStock()->limit(6);
                            }])
                            ->orderBy('sort_order', 'asc')
                            ->get();

        return view('deals.index', compact('deals', 'featuredDeals'));
    }

    /**
     * Display a specific deal.
     */
    public function show($slug)
    {
        $deal = Deal::where('slug', $slug)
                    ->with(['products' => function($query) {
                        $query->active()->inStock();
                    }])
                    ->firstOrFail();

        // Check if deal is active
        if (!$deal->isActive()) {
            return redirect()->route('deals.index')
                            ->with('warning', 'This deal has expired or is no longer available.');
        }

        return view('deals.show', compact('deal'));
    }

    /**
     * Apply deal to cart (if needed for special deals)
     */
    public function apply(Request $request, $id)
    {
        $deal = Deal::active()->findOrFail($id);

        // Check if usage limit reached
        if ($deal->isUsageLimitReached()) {
            return back()->with('error', 'This deal has reached its usage limit.');
        }

        // Store deal in session
        session(['applied_deal' => $deal->id]);

        // Increment usage count
        $deal->incrementUsage();

        return back()->with('success', 'Deal applied successfully!');
    }

    /**
     * Remove deal from cart
     */
    public function remove()
    {
        session()->forget('applied_deal');
        return back()->with('success', 'Deal removed successfully.');
    }

    /**
     * ADMIN METHODS - For admin panel
     */

    /**
     * Display admin deals index.
     */
    public function adminIndex()
    {
        $deals = Deal::orderBy('created_at', 'desc')->paginate(20);
        return view('admin.deals.index', compact('deals'));
    }

    /**
     * Show form to create new deal.
     */
    public function create()
    {
        $products = Product::active()->inStock()->get();
        return view('admin.deals.create', compact('products'));
    }

    /**
     * Store new deal.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'type' => 'required|in:percentage,fixed,bundle,special',
            'discount_value' => 'required_if:type,percentage,fixed|numeric|min:0',
            'min_purchase' => 'nullable|numeric|min:0',
            'usage_limit' => 'nullable|integer|min:1',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
            'is_active' => 'boolean',
            'featured' => 'boolean',
            'sort_order' => 'nullable|integer',
            'image' => 'nullable|image|max:2048',
            'products' => 'nullable|array',
            'products.*' => 'exists:products,id'
        ]);

        // Handle image upload
        if ($request->hasFile('image')) {
            $path = $request->file('image')->store('deals', 'public');
            $validated['image'] = $path;
        }

        // Generate unique slug
        $slug = Str::slug($validated['title']);
        $originalSlug = $slug;
        $counter = 1;

        // Check if slug exists and make it unique
        while (Deal::where('slug', $slug)->exists()) {
            $slug = $originalSlug . '-' . $counter;
            $counter++;
        }

        // Add slug to validated data
        $validated['slug'] = $slug;

        // Create deal
        $deal = Deal::create($validated);

        // Attach products if provided
        if ($request->has('products')) {
            $deal->products()->attach($request->products);
        }

        // Send notification to all users (queued)
        if ($deal->is_active) {
            SendDealNotificationToAllUsers::dispatch($deal)
                ->delay(now()->addSeconds(10));
        }

        return redirect()->route('admin.deals.index')
                        ->with('success', 'Deal created successfully! ' . 
                               ($deal->is_active ? 'Notification sent to all users.' : ''));
    }

    /**
     * Show form to edit deal.
     */
    public function edit(Deal $deal)
    {
        $products = Product::active()->inStock()->get();
        $selectedProducts = $deal->products->pluck('id')->toArray();
        
        return view('admin.deals.edit', compact('deal', 'products', 'selectedProducts'));
    }

    /**
     * Update deal.
     */
    public function update(Request $request, Deal $deal)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'type' => 'required|in:percentage,fixed,bundle,special',
            'discount_value' => 'required_if:type,percentage,fixed|numeric|min:0',
            'min_purchase' => 'nullable|numeric|min:0',
            'usage_limit' => 'nullable|integer|min:1',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
            'is_active' => 'boolean',
            'featured' => 'boolean',
            'sort_order' => 'nullable|integer',
            'image' => 'nullable|image|max:2048',
            'products' => 'nullable|array',
            'products.*' => 'exists:products,id'
        ]);

        // Get old status before update
        $wasInactive = !$deal->is_active;

        // Generate slug if title changed
        if ($request->title !== $deal->title) {
            $slug = Str::slug($validated['title']);
            $originalSlug = $slug;
            $counter = 1;

            while (Deal::where('slug', $slug)->where('id', '!=', $deal->id)->exists()) {
                $slug = $originalSlug . '-' . $counter;
                $counter++;
            }

            $validated['slug'] = $slug;
        }

        // Handle image upload
        if ($request->hasFile('image')) {
            // Delete old image
            if ($deal->image) {
                Storage::disk('public')->delete($deal->image);
            }
            
            $path = $request->file('image')->store('deals', 'public');
            $validated['image'] = $path;
        } elseif ($request->has('remove_image')) {
            // Remove image if requested
            if ($deal->image) {
                Storage::disk('public')->delete($deal->image);
                $validated['image'] = null;
            }
        }

        // Update deal
        $deal->update($validated);

        // Sync products
        if ($request->has('products')) {
            $deal->products()->sync($request->products);
        } else {
            $deal->products()->detach();
        }

        // Send notification if deal was just activated
        if ($wasInactive && $deal->is_active) {
            SendDealNotificationToAllUsers::dispatch($deal)
                ->delay(now()->addSeconds(10));
        }

        return redirect()->route('admin.deals.index')
                        ->with('success', 'Deal updated successfully!' . 
                               ($wasInactive && $deal->is_active ? ' Notification sent to all users.' : ''));
    }

    /**
     * Delete deal.
     */
    public function destroy(Deal $deal)
    {
        // Delete image if exists
        if ($deal->image) {
            Storage::disk('public')->delete($deal->image);
        }

        // Detach products
        $deal->products()->detach();

        // Delete deal
        $deal->delete();

        return redirect()->route('admin.deals.index')
                        ->with('success', 'Deal deleted successfully!');
    }

    /**
     * Toggle deal status.
     */
    public function toggleStatus(Deal $deal)
    {
        $wasInactive = !$deal->is_active;
        
        $deal->update(['is_active' => !$deal->is_active]);
        
        $status = $deal->is_active ? 'activated' : 'deactivated';
        
        // Send notification if deal was just activated
        if ($wasInactive && $deal->is_active) {
            SendDealNotificationToAllUsers::dispatch($deal)
                ->delay(now()->addSeconds(10));
        }
        
        return back()->with('success', "Deal {$status} successfully!" . 
                       ($wasInactive && $deal->is_active ? ' Notification sent to all users.' : ''));
    }
}